package internal

import (
	"context"
	"math"
	"math/rand"
	"sync"
	"time"

	"a.yandex-team.ru/library/go/yandex/blackbox"
	"a.yandex-team.ru/library/go/yandex/blackbox/httpbb"
	"a.yandex-team.ru/library/go/yandex/tvm"
	"a.yandex-team.ru/library/go/yandex/tvm/tvmtool"
	"a.yandex-team.ru/passport/backend/userdata/yt_schema/common"
	passportBlackbox "a.yandex-team.ru/passport/shared/golibs/blackbox"
	"a.yandex-team.ru/passport/shared/golibs/logger"
)

type token struct{}

const KinopoiskUIDBoundary = 1110000000000000
const PddUIDBoundary = 1130000000000000

func InitBlackboxChannel() chan uint64 {
	c := make(chan uint64)
	return c
}

func NewBlackboxClient(cfg Config) passportBlackbox.Client {
	tvmClient, err := tvmtool.NewClient(
		cfg.TVMAPIURL,
		tvmtool.WithSrc(cfg.TVMSource),
		tvmtool.WithAuthToken(cfg.TVMAuthToken),
	)
	if err != nil {
		panic(err)
	}

	bbClient, err := passportBlackbox.NewBlackBoxClient(
		httpbb.Environment{
			BlackboxHost: cfg.BlackboxURL,
			TvmID:        tvm.ClientID(cfg.BlackboxDestination),
		},
		httpbb.WithTimeout(cfg.BlackboxTimeout),
		httpbb.WithTVM(tvmClient),
	)

	if err != nil {
		panic(err)
	}

	return bbClient
}

func BlackboxGetMaxUID(bbClient passportBlackbox.Client) (*passportBlackbox.GetMaxUIDResponse, error) {
	response, err := bbClient.GetMaxUID(context.Background())
	if err == nil {
		return response, nil
	} else {
		return nil, err
	}
}

func BlackboxUserinfo(unistat *UnistatServer, currentIP string, bbClient passportBlackbox.Client, UIDs []uint64) ([]blackbox.User, error) {
	request := blackbox.UserInfoRequest{
		RegName:         true,
		UIDs:            UIDs,
		UserIP:          currentIP,
		Aliases:         common.AliasesToQuery,
		Attributes:      common.AttributesToQuery,
		DBFields:        common.DBFieldsToQuery,
		GetPublicID:     true,
		GetPublicName:   true,
		GetFamilyInfo:   true,
		GetPhones:       blackbox.GetPhonesBound,
		PhoneAttributes: common.PhoneAttributesToQuery,
	}
	beforeRequest := time.Now()
	response, err := bbClient.UserInfo(context.Background(), request)
	duration := float64(time.Since(beforeRequest) / time.Millisecond)
	unistat.BlackboxDuration.Update(duration)
	unistat.BlackboxTotal.Update(1)
	if err == nil {
		unistat.BlackboxSuccess.Update(1)
		return response.Users, nil
	} else {
		unistat.BlackboxErrors.Update(1)
		return nil, err
	}
}

func UIDsDownloader(
	unistat *UnistatServer,
	currentIP string,
	bbClient passportBlackbox.Client,
	UIDsChannel chan []uint64,
	exportChannel chan []blackbox.User,
	wg *sync.WaitGroup) {

	for UIDs := range UIDsChannel {
		pause := 1
		pauseMultiplier := 2
		maxPause := rand.Intn(60)
		attempts := 0
		for {
			attempts += 1
			users, err := BlackboxUserinfo(unistat, currentIP, bbClient, UIDs)
			if err == nil {
				if len(users) > 0 {
					exportChannel <- users
				}
				break
			} else {
				interval := math.Min(float64(pause*pauseMultiplier*attempts), float64(maxPause))
				sleepDuration := time.Duration(interval) * time.Second
				logger.Log().Warnf("Attempt #%+v; %+v; next attempt in %+v; UIDs %+v", attempts, err, sleepDuration, UIDs)
				time.Sleep(sleepDuration)
			}
		}
	}
	wg.Done()
}

func JoinPipeline(
	unistat *UnistatServer,
	cfg Config,
	UIDsPerUserinfo int,
	workersCount int,
	maxUID uint64,
	maxPDDUID uint64,
	uidsChannel chan uint64,
	exportChannel chan []blackbox.User) {

	currentIP := GetCurrentIPAddress()

	bbClient := NewBlackboxClient(cfg)
	var UIDs []uint64

	downloadQueue := make(chan []uint64)
	var wg sync.WaitGroup

	wg.Add(workersCount)
	for i := 0; i < workersCount; i++ {
		go UIDsDownloader(
			unistat,
			currentIP,
			bbClient,
			downloadQueue,
			exportChannel,
			&wg,
		)
	}

	var maxSeenNormalUID uint64 = 0
	var maxSeenPDDUID uint64 = PddUIDBoundary

	for uid := range uidsChannel {
		if uid >= KinopoiskUIDBoundary && uid < PddUIDBoundary {
			// кинопоисковых пользователей пропускаем, они совсем не интересны
			continue
		}
		if uid > maxSeenNormalUID && uid < PddUIDBoundary {
			maxSeenNormalUID = uid
		}
		if uid > PddUIDBoundary && uid > maxSeenPDDUID {
			maxSeenPDDUID = uid
		}

		UIDs = append(UIDs, uid)
		if len(UIDs) >= UIDsPerUserinfo {
			downloadQueue <- UIDs
			UIDs = nil
		}
	}
	if len(UIDs) > 0 {
		downloadQueue <- UIDs
		UIDs = nil
	}
	// скачали userinfo по всем уидам из входного файла
	// теперь время скачать тех, кто зарегистрировался после формирования файла

	// нормальные пользователи
	if maxUID > 0 {
		for uid := maxSeenNormalUID + 1; uid <= maxUID; uid += 1 {
			UIDs = append(UIDs, uid)
			if len(UIDs) >= UIDsPerUserinfo {
				downloadQueue <- UIDs
				UIDs = nil
			}
		}
		if len(UIDs) > 0 {
			downloadQueue <- UIDs
			UIDs = nil
		}
	}
	// ПДД пользователи
	if maxPDDUID > 0 {
		for uid := maxSeenPDDUID + 1; uid <= maxPDDUID; uid += 1 {
			UIDs = append(UIDs, uid)
			if len(UIDs) >= UIDsPerUserinfo {
				downloadQueue <- UIDs
				UIDs = nil
			}
		}
		if len(UIDs) > 0 {
			downloadQueue <- UIDs
		}
	}
	logger.Log().Debugf("Closing download queue")
	close(downloadQueue)

	// ждём завершения воркеров, скачивающих уже существующие уиды
	logger.Log().Debugf("Waiting for workers to finish download")
	wg.Wait()
	logger.Log().Debugf("Finished downloading all previously known UIDs")

	logger.Log().Debugf("Closing exportChannel")
	close(exportChannel)
}
