package internal

import (
	"bufio"
	"compress/gzip"
	"os"
	"strconv"
	"strings"
	"sync"
	"sync/atomic"
	"time"

	"a.yandex-team.ru/library/go/yandex/blackbox"
	"a.yandex-team.ru/passport/backend/userdata/yt_schema/common"
	"a.yandex-team.ru/passport/shared/golibs/logger"
)

func check(err error) {
	if err != nil {
		panic(err)
	}
}

func InitExportChannel() chan []blackbox.User {
	c := make(chan []blackbox.User)
	return c
}

func intAttribute(value string) *int64 {
	rawAttributeValue := value
	parsedAsInt, err := strconv.ParseInt(rawAttributeValue, 10, 64)
	if err != nil {
		return nil
	}
	return &parsedAsInt
}

func DBFieldsToSIDs(dbFields map[blackbox.UserDBField]string) []int64 {
	var SIDs []int64
	for name, value := range dbFields {
		if value == "" {
			continue
		}
		bits := strings.Split(string(name), ".")
		sid, err := strconv.ParseInt(bits[len(bits)-1], 10, 64)
		if err != nil {
			panic(err)
		}
		SIDs = append(SIDs, sid)
	}
	return SIDs
}

func ParseAndExport(user blackbox.User) common.ExportEntry {
	uid := strconv.FormatUint(user.UID.ID, 10)
	login := user.Attributes[("1008")]
	aliases := []int64{}
	for k := range user.Aliases {
		alias, _ := strconv.ParseInt(string(k), 10, 64)
		aliases = append(aliases, alias)
	}

	var familyID *string
	var securePhoneNumber *string
	var bankPhoneNumber *string
	securePhoneNumber = nil
	bankPhoneNumber = nil
	allPhoneNumbers := []string{}
	var birthday *string
	birthday = nil

	displayName := user.DisplayName.Name
	publicID := user.PublicID

	if user.Attributes[blackbox.UserAttributePersonBirthday] != "" {
		bd := user.Attributes[blackbox.UserAttributePersonBirthday]
		birthday = &bd
	}
	regDate := intAttribute(user.Attributes[blackbox.UserAttributeAccountRegistrationDatetime])
	hasPlus := user.Attributes[blackbox.UserAttributeAccountHavePlus] != ""
	hasPlusCashback := user.Attributes[("1025")] != ""
	securePhoneID := intAttribute(user.Attributes[("110")])
	accountEnabled := user.Attributes[("3")] == ""
	defaultAvatar := user.Attributes[blackbox.UserAttributeAvatarDefault]
	firstName := user.Attributes[blackbox.UserAttributePersonFirstname]
	lastName := user.Attributes[blackbox.UserAttributePersonLastname]
	userDefinedLogin := user.Attributes[("2")]
	passwordUpdatedTimestamp := intAttribute(user.Attributes[blackbox.UserAttributePasswordUpdateDatetime])
	passwordQuality := intAttribute(user.Attributes[blackbox.UserAttributePasswordQuality])
	passwordChangingReason := intAttribute(user.Attributes[blackbox.UserAttributePasswordForcedChangingReason])
	hasAppPassword := user.Attributes[blackbox.UserAttributePasswordWebOnlyaccountEnableAppPassword] != ""
	country := user.Attributes[blackbox.UserAttributePersonCountry]
	language := user.Attributes[blackbox.UserAttributePersonLanguage]
	gender := user.Attributes[blackbox.UserAttributePersonGender]
	showFIOInPublicName := user.Attributes[("185")] != ""
	passwordIsChangingRequiredTimestamp := intAttribute(user.Attributes[("195")])
	SIDs := DBFieldsToSIDs(user.DBFields)
	sms2fa := user.Attributes[("200")]
	publicName := user.DisplayName.PublicName
	if user.FamilyInfo.FamilyID != "" {
		familyID = &user.FamilyInfo.FamilyID
	}

	for _, phone := range user.PhoneList {
		if phone.IsSecured {
			securePhoneNumber = &phone.E164Number
		}
		if phone.IsBankPhoneNumber {
			bankPhoneNumber = &phone.E164Number
		}
		allPhoneNumbers = append(allPhoneNumbers, phone.E164Number)
	}

	return common.ExportEntry{
		UID:                                 uid,
		Login:                               login,
		Aliases:                             aliases,
		Birthday:                            birthday,
		HasBirthday:                         birthday != nil,
		HasPlus:                             hasPlus,
		HasPlusCashback:                     hasPlusCashback,
		SecurePhoneID:                       securePhoneID,
		SecurePhoneNumber:                   securePhoneNumber,
		BankPhoneNumber:                     bankPhoneNumber,
		HasSecurePhoneID:                    securePhoneID != nil,
		AccountEnabled:                      accountEnabled,
		DefaultAvatar:                       &defaultAvatar,
		HasDefaultAvatar:                    defaultAvatar != "",
		FirstName:                           &firstName,
		HasFirstName:                        firstName != "",
		LastName:                            &lastName,
		HasLastName:                         lastName != "",
		UserDefinedLogin:                    &userDefinedLogin,
		DisplayName:                         &displayName,
		HasDisplayName:                      user.DisplayName.Name != "",
		HasPassword:                         user.HasPassword,
		HasHint:                             user.HasHint,
		PasswordQuality:                     passwordQuality,
		PasswordChangingReason:              passwordChangingReason,
		PasswordUpdatedTimestamp:            passwordUpdatedTimestamp,
		PhoneNumbers:                        allPhoneNumbers,
		RegDate:                             regDate,
		HasAppPassword:                      hasAppPassword,
		Country:                             &country,
		Language:                            &language,
		Gender:                              &gender,
		ShowFIOInPublicName:                 showFIOInPublicName,
		SID:                                 SIDs,
		PublicID:                            &publicID,
		PublicName:                          &publicName,
		PasswordIsChangingRequiredTimestamp: passwordIsChangingRequiredTimestamp,
		SMS2FA:                              sms2fa != "",
		FamilyID:                            familyID,
		Karma:                               user.KarmaStatus,
	}
}

func worker(cin chan []blackbox.User, cout chan []byte, wg *sync.WaitGroup) {
	defer wg.Done()

	for users := range cin {
		for _, user := range users {
			ee := ParseAndExport(user)
			eeMarshalled, _ := common.EncodeExportEntry(ee)
			eeMarshalled = append(eeMarshalled, 10)
			cout <- eeMarshalled
		}
	}
}

func dumper(unistat *UnistatServer, outputFile string, buffSize int, cin chan []byte, dumpToStdout bool, gzipDump bool, gzipLevel int, counter *uint64, wg *sync.WaitGroup) {
	defer wg.Done()

	var f *os.File
	if dumpToStdout {
		f = os.Stdout
	} else {
		f2, err := os.OpenFile(outputFile, os.O_APPEND|os.O_CREATE|os.O_WRONLY, 0644)
		f = f2
		check(err)
	}
	if gzipDump {
		var writer *gzip.Writer
		writer, err := gzip.NewWriterLevel(f, gzipLevel)
		check(err)
		defer func() {
			check(writer.Flush())
			check(writer.Close())
		}()

		for user := range cin {
			unistat.Exported.Update(1)
			atomic.AddUint64(counter, 1)
			_, err := writer.Write(user)
			check(err)
		}
	} else {
		writer := bufio.NewWriterSize(f, buffSize)
		defer func() {
			err := writer.Flush()
			check(err)
		}()

		for user := range cin {
			unistat.Exported.Update(1)
			atomic.AddUint64(counter, 1)
			_, err := writer.Write(user)
			check(err)
		}
	}
}

func ExportingChannel(unistat *UnistatServer, outputFile string, c chan []blackbox.User, workers int, dumpingWorkers int, dumpToStdout bool, gzipDump bool, gzipLevel int, buffSize int, done chan int) {
	var counter uint64
	go func() {
		for range time.Tick(10 * time.Second) {
			logger.Log().Debugf("Downloaded %d users...", counter)
		}
	}()

	var wgSerializers sync.WaitGroup
	var wgDumpers sync.WaitGroup
	processed := make(chan []byte)

	for i := 0; i < workers; i++ {
		wgSerializers.Add(1)
		go worker(c, processed, &wgSerializers)
	}

	for i := 0; i < dumpingWorkers; i++ {
		wgDumpers.Add(1)
		go dumper(
			unistat,
			outputFile+strconv.Itoa(i),
			buffSize,
			processed,
			dumpToStdout,
			gzipDump,
			gzipLevel,
			&counter,
			&wgDumpers,
		)
	}

	logger.Log().Debugf("Waiting for exporting workers to finish")
	wgSerializers.Wait()
	logger.Log().Debugf("Closing exporting queue")
	close(processed)
	wgDumpers.Wait()
	logger.Log().Infof("Exported %d UIDs in total", counter)
	logger.Log().Infof("Done exporting!")
	done <- 1
}
