package internal

import (
	"context"
	"net"
	"net/http"
	"os"
	"os/signal"
	"sync/atomic"
	"syscall"
	"time"

	"github.com/go-chi/chi/v5"

	"a.yandex-team.ru/passport/shared/golibs/logger"
)

var (
	healthy int32
)

func InitUnistatServer(listenAddr string, listenNetwork string, unistatServer *UnistatServer) {
	logger.Log().Debug("Initializing web app")

	r := chi.NewRouter()

	r.Get("/unistat", unistatServer.HandlerUnistat(logger.Log()))

	server := &http.Server{
		Addr:         listenAddr,
		Handler:      r,
		ReadTimeout:  10 * time.Second,
		WriteTimeout: 10 * time.Second,
		IdleTimeout:  10 * time.Second,
	}

	done := make(chan bool)
	quit := make(chan os.Signal, 1)

	signal.Notify(quit, os.Interrupt, syscall.SIGTERM)

	go func() {
		<-quit
		logger.Log().Info("Server is shutting down...")
		atomic.StoreInt32(&healthy, 0)

		ctx, cancel := context.WithTimeout(context.Background(), 30*time.Second)
		defer cancel()

		server.SetKeepAlivesEnabled(false)
		if err := server.Shutdown(ctx); err != nil {
			logger.Log().Fatalf("Could not gracefully shutdown the server: %v\n", err)
		}
		close(done)
	}()

	logger.Log().Infof("Server is ready to handle requests at %+v", listenAddr)
	atomic.StoreInt32(&healthy, 1)

	listener, err := net.Listen(listenNetwork, listenAddr)
	if err != nil && err != http.ErrServerClosed {
		logger.Log().Fatalf("Could not listen on %s (network %s): %v\n", listenAddr, listenNetwork, err)
	}
	if err := server.Serve(listener); err != nil {
		logger.Log().Fatalf("Could not serve: %v\n", err)
	}
	<-done
	logger.Log().Info("Server stopped")
}
