import argparse
from datetime import (
    datetime,
    timedelta,
)
import os
import sys

from nile.api.v1 import (
    extractors as ne,
    filters as nf,
)
from passport.backend.clients.mr import NileYtClient
from qb2.api.v1.typing import (
    String,
    UInt64,
)
import yenv


def get_cluster():
    token = os.environ.get('YT_TOKEN')
    pool = os.environ.get('YT_POOL')
    cluster = NileYtClient(
        proxy='hahn.yt.yandex.net',
        token=token,
        pool=pool,
        spec=dict(),
    )
    return cluster


def main():
    parser = argparse.ArgumentParser()
    subparsers = parser.add_subparsers()

    parser_find_alive = subparsers.add_parser('find_alive_uids', help='Find UIDs that are not deleted')
    parser_find_alive.add_argument('date', metavar='DATE')
    parser_find_alive.set_defaults(func=find_alive_uids)

    parser_find_alive_old = subparsers.add_parser(
        'find_alive_uids_old', help='Find UIDs that are not deleted in old table'
    )
    parser_find_alive_old.add_argument('date', metavar='DATE')
    parser_find_alive_old.set_defaults(func=find_alive_uids_old_userdata)

    parser_find_updated_uids = subparsers.add_parser('find_updated_uids', help='Find updated UIDs for specified date')
    parser_find_updated_uids.add_argument('date', metavar='DATE')
    parser_find_updated_uids.set_defaults(func=find_updated_uids)

    parser_merge = subparsers.add_parser('merge_old_with_new', help='Merge old userdata with a new delta')
    parser_merge.add_argument('date', metavar='DATE')
    parser_merge.set_defaults(func=merge_old_with_new)

    if len(sys.argv) <= 1:
        sys.argv.append('--help')

    options = parser.parse_args()
    options.func(options.date)


def find_alive_uids_old_userdata(date):
    cluster = get_cluster()

    prev_day = (datetime.strptime(date, '%Y-%m-%d') - timedelta(days=1)).strftime('%Y-%m-%d')
    print(prev_day)

    job = cluster.job('find undeleted uids').env(
        templates={
            'date': prev_day,
        },
    )

    job.table('//statbox/heavy-dict/passport_userdata/$date').project('uid', 'account_status',).filter(
        nf.not_(nf.equals('account_status', 'enabled')),
    ).project(
        uid=ne.custom(int, 'uid'),
    ).put(
        '//tmp/undeleted-uids',
        compression_level='heaviest',
        merge_strategy='always',
        schema={'uid': UInt64},
    )

    job.run()


def find_alive_uids(date):
    cluster = get_cluster()

    prev_day = (datetime.strptime(date, '%Y-%m-%d') - timedelta(days=1)).strftime('%Y-%m-%d')
    print(prev_day)

    job = cluster.job('find undeleted uids').env(
        templates={
            'date': prev_day,
        },
    )

    job.table('//home/passport/{yenv_type}/userdata/$date'.format(yenv_type=yenv.type,)).project('uid',).project(
        uid=ne.custom(int, 'uid'),
    ).put(
        '//tmp/undeleted-uids',
        compression_level='heaviest',
        merge_strategy='always',
        schema={'uid': UInt64},
    )

    job.run()


def find_updated_uids(date):
    cluster = get_cluster()

    job = cluster.job('find undeleted uids').env(
        templates={
            'date': date,
        },
    )

    job.table('//logs/passport-log/1d/$date').unique(
        'uid',
        keep_only_group_fields=True,
    ).filter(
        nf.and_(
            nf.not_(nf.equals('uid', None)),
            nf.custom(lambda uid: uid.decode('utf-8').isdigit(), 'uid'),
        ),
        intensity='cpu',
    ).put(
        '//home/passport/{yenv_type}/userdata/tmp/$date-delta-uids'.format(
            yenv_type=yenv.type,
        ),
        compression_level='normal',
        merge_strategy='always',
        schema={'uid': String},
    )

    job.run()


def merge_old_with_new(date):
    cluster = get_cluster()
    prev_day = (datetime.strptime(date, '%Y-%m-%d') - timedelta(days=1)).strftime('%Y-%m-%d')
    job = cluster.job('merge old with new').env(
        templates={
            'date': date,
            'prev_day': prev_day,
        },
    )

    job.table('//home/passport/{yenv_type}/userdata/$prev_day'.format(yenv_type=yenv.type,)).join(
        job.table(
            '//home/passport/{yenv_type}/userdata/tmp/$date-updated'.format(
                yenv_type=yenv.type,
            )
        ),
        type='left_only',
        by='uid',
        assume_unique=True,
        allow_undefined_keys=False,
        assume_defined=True,
    ).concat(
        job.table('//home/passport/{yenv_type}/userdata/tmp/$date-updated'.format(yenv_type=yenv.type,)).project(
            ne.all(),
            intensity='cpu',
        ),
    ).put(
        '//home/passport/{yenv_type}/userdata/tmp/$date-finished'.format(
            yenv_type=yenv.type,
        ),
        compression_level='heaviest',
        merge_strategy='always',
    )

    job.run()


if __name__ == '__main__':
    main()
