# -*- coding: utf-8 -*-

import base64
import collections
from functools import partial
from inspect import isclass
import operator
from os import path
import random

import netaddr
from passport.backend.utils.string import (
    smart_bytes,
    smart_text,
)
from passport.backend.utils.string import smart_str  # noqa
import six
from six import (
    iteritems,
    StringIO,
)
from six.moves import (
    collections_abc,
    range,
)
from six.moves.urllib.parse import (
    quote,
    unquote,
    urlsplit,
    urlunsplit,
)


try:
    from io import TextIOWrapper as file  # for py3 compatibility
except ImportError:
    pass


class ClassMapping(collections_abc.Mapping):
    """
    Упорядоченное неизменяемое отображение ключи которого -- это классы.

    В конструктор нужно передать список пар (класс-ключ, значение).
    """
    def __init__(self, relations):
        self._relations = collections.OrderedDict(relations)

    def __len__(self):
        return len(self._relations)

    def __getitem__(self, _class):
        # Проверим, что _class действительно класс
        if not isclass(_class):
            raise KeyError(_class)

        try:
            return self._relations[_class]
        except KeyError:
            pass
        for domain_class in self._relations:
            if issubclass(_class, domain_class):
                value = self._relations[domain_class]
                break
        else:
            raise KeyError(_class)
        return value

    def __iter__(self):
        for _class in self._relations:
            yield _class


class ClassPropertyDescriptor(object):
    def __init__(self, fget, fset=None):
        self.fget = fget
        self.fset = fset

    def __get__(self, obj, klass=None):
        if klass is None:
            klass = type(obj)
        return self.fget.__get__(obj, klass)()

    def __set__(self, obj, value):
        if not self.fset:
            raise AttributeError("can't set attribute")
        type_ = type(obj)
        return self.fset.__get__(obj, type_)(value)

    def setter(self, func):
        if not isinstance(func, (classmethod, staticmethod)):
            func = classmethod(func)
        self.fset = func
        return self


def _chop_file(tail, length):
    head = []
    for line in tail:
        line = line.strip('\n')
        head.append(line)
        if len(head) >= length:
            yield head
            head = []
    if head:
        yield head


def _chop_list(tail, length):
    while tail:
        head, tail = tail[:length], tail[length:]
        yield head


def _chop_iterable(tail, length):
    it = iter(tail)
    while True:
        chunk = list()
        try:
            for _ in range(length):
                chunk.append(next(it))
        except StopIteration:
            pass
        if not chunk:
            break
        yield chunk


def _generate_random_code(length):
    if length <= 0:
        raise ValueError("'length' must be positive integer")
    start = 10 ** (length - 1)
    stop = 10 ** length
    result = random.SystemRandom().randrange(start, stop)
    return str(result)


def _merge_dicts_couple(dict1, dict2):
    ret = dict(dict1)
    ret.update(dict2)
    common_keys = set(dict1.keys()) & set(dict2.keys())
    for key in common_keys:
        merge = _MERGE_OPS.get(type(dict1[key]))
        if merge:
            ret[key] = merge(dict1[key], dict2[key])
    return ret


_MERGE_OPS = ClassMapping([
    (dict, _merge_dicts_couple),
    (list, operator.add),
    (set, operator.or_),
])


def chop(tail, length):
    if isinstance(tail, (file, StringIO)):
        return _chop_file(tail, length)
    elif hasattr(tail, '__getitem__'):
        return _chop_list(tail, length)
    else:
        return _chop_iterable(tail, length)


def chunks(l, n):
    for i in range(0, len(l), n):
        yield l[i:i + n]


def classproperty(func):
    if not isinstance(func, (classmethod, staticmethod)):
        func = classmethod(func)

    return ClassPropertyDescriptor(func)


class context_manager_to_decorator(object):
    """
    Преобразует контекстный менеджер в декоратор.

    d = context_manager_to_decorator(c)
    df = d(f)

    тогда вызов

    df(1, 2)

    равносилен

    with c():
        f(1, 2)
    """
    def __init__(self, context_manager):
        self._context_manager = context_manager

    def __call__(self, f):
        def wrapper(*args, **kwargs):
            with self._context_manager():
                return f(*args, **kwargs)
        return wrapper


def cut_host(host, count):
    return '.'.join(host.split('.')[-1 * count:])


def deep_merge(*dicts):
    ret = {}
    for dict_ in dicts:
        ret = _merge_dicts_couple(ret, dict_)
    return ret


def encode_query_mapping(query):
    if isinstance(query, dict):
        query_was_dict = True
        query = query.items()
    else:
        query_was_dict = False

    encoded_query = [(smart_bytes(arg), smart_bytes(val)) for arg, val in query]

    if query_was_dict:
        encoded_query = dict(encoded_query)
    return encoded_query


def filter_none(collection):
    if isinstance(collection, (list, tuple)):
        return list(filter(lambda x: x is not None, collection))


def flatten(d,  sep='.', prefix=''):
    items = []
    for k, v in d.items():
        if isinstance(v, collections_abc.MutableMapping):
            items.extend(flatten(v, sep=sep, prefix=prefix + k + sep).items())
        else:
            items.append((prefix + k, v))
    return dict(items)


def flatten_values(values):
    """
    Коллекции по ключу разбивает на отдельные ключ-значение
    :rtype: list
    """
    result = []

    def update_values(it):
        for key, value in it:
            if isinstance(value, (list, set, tuple)):
                result.extend([(key, v) for v in value])
            else:
                result.append((key, value))

    if hasattr(values, 'items'):
        update_values(iteritems(values))
    else:
        update_values(values)
    return result


def float_or_default(value, default=None):
    try:
        return float(value)
    except (TypeError, ValueError):
        return default


def from_base64_standard(text):
    # https://stackoverflow.com/a/49459036
    return base64.standard_b64decode(six.ensure_binary(text) + six.ensure_binary('=='))


def from_base64_url(text):
    if len(text) % 4:
        text += b'=' * (4 - len(text) % 4)
    return base64.urlsafe_b64decode(six.ensure_binary(text))


def url_to_ascii(url):
    parsed = urlsplit(smart_text(url))

    login_and_password, at, host_and_port = parsed.netloc.rpartition('@')
    login, colon1, password = login_and_password.partition(':')
    host, colon2, port = host_and_port.partition(':')

    scheme = parsed.scheme
    login = quote(login.encode('utf8'))
    password = quote(password.encode('utf8'))
    host = host.encode('idna').decode('utf8')
    path = '/'.join(
        quote(
            unquote(pce).encode('utf8'),
            safe='',  # квотим и слеши тоже
        )
        for pce in parsed.path.split('/')
    )
    query = quote(unquote(parsed.query).encode('utf8'), safe='=&?/')
    fragment = quote(unquote(parsed.fragment).encode('utf8'))

    netloc = ''.join((login, colon1, password, at, host, colon2, port))
    return urlunsplit((scheme, netloc, path, query, fragment))


def bytes_to_hex(bytes_):
    if six.PY3:
        return bytes_.hex()
    else:
        return bytes_.encode('hex')


def generate_random_code(length):
    # Делегирую для того, чтобы нужно было патчить только в одном месте.
    return _generate_random_code(length)


def format_code_by_3(code, delimiter=' '):
    """Форматирование кода: по три символа через delimiter"""
    code_bits = []
    for i in range(0, len(code), 3):
        code_bits.append(code[i:i + 3])
    return delimiter.join(code_bits)


def normalize_code(code, delimiters=' -'):
    """Удаляет разделители из кода, если они есть"""
    delimiters = set(delimiters)
    return ''.join(c for c in smart_text(code) if c not in delimiters)


def get_default_if_none(d, key, default):
    # если в словаре значение ключа отсутствует или None, то возвращает default
    value = d.get(key)
    if value is None:
        return default
    else:
        return value


def identity(x):
    return x


def inheritors(klass):
    subclasses = set()
    work = [klass]
    while work:
        parent = work.pop()
        for child in parent.__subclasses__():
            if child not in subclasses:
                subclasses.add(child)
                work.append(child)
    return subclasses


def int_or_default(value, default=None):
    try:
        return int(value)
    except (TypeError, ValueError):
        return default


def map_dict(dictionary, map_table):
    """
    Пребразует значения на ключах словаря _dict с помощью функций
    на соответствующих ключах словаря map_table.
    """
    return {key: map_table.get(key, identity)(dictionary[key])
            for key in dictionary}


class method_decorator(object):
    """Преобразует декоратор функции в декоратор метода."""
    def __init__(self, decorate):
        self._decorate = decorate

    def __call__(self, unbound_method):
        this = self

        def wrapper(self, *args, **kwargs):
            bound_method = partial(unbound_method, self)
            bound_method.__name__ = unbound_method.__name__
            decorated_bound_method = this._decorate(bound_method)
            return decorated_bound_method(*args, **kwargs)

        return wrapper


def merge_dicts(d1, *args):
    merged = dict(d1)
    for d in args:
        merged.update(d)
    return merged


def next_greater_power_of_2(x):
    return 2 ** (x - 1).bit_length()


def noneless_dict(*args, **kwargs):
    return remove_none_values(dict(*args, **kwargs))


def path_exists(full_path):
    # Для правильного подсчета покрытия,
    # т.к. coverage внутри себя во время тестов вызывает методы модуля os.path
    # По мотивам: https://bitbucket.org/ned/coveragepy/issues/518/coverage-fails-with-mocked-ospathislink
    return path.exists(full_path)   # pragma: no cover


def remove_none_values(d):
    if hasattr(d, 'items'):
        retval = {k: v for k, v in iteritems(d) if v is not None}
    else:
        retval = [(k, v) for k, v in d if v is not None]
    return retval


def smart_merge_dicts(destination, source, list_policy='merge', copy=True):
    if copy:
        destination = dict(destination)

    for key, value in source.items():
        if isinstance(value, dict):
            policy = value.pop('_policy', 'merge')
            if policy == 'override':
                destination[key] = value
            elif policy == 'bypass':
                if key not in destination:
                    destination[key] = value
            elif policy == 'merge':
                destination.setdefault(key, {})
                destination[key] = smart_merge_dicts(destination[key], value, list_policy=list_policy)
        elif isinstance(value, (tuple, list)) and list_policy == 'merge':
            destination.setdefault(key, [])
            if isinstance(destination[key], (tuple, list)):
                destination[key] = [_ for _ in destination[key]]
                destination[key].extend(value)
            else:
                destination[key] = value
        else:
            destination[key] = value

    return destination


def sorted_noneless_dict(d):
    return collections.OrderedDict((
        (k, v)
        for k, v in sorted(d.items(), key=lambda x: x[0])
        if v is not None
    ))


def to_base64_standard(text):
    return base64.standard_b64encode(text).strip(b'=')


def to_base64_url(text):
    return base64.urlsafe_b64encode(text).strip(b'=')


def truncate_timestamp(unixtime, seconds):
    return unixtime - (unixtime % seconds)


def unflatten(flat_dict, sep='.'):
    unflatted_dict = {}
    for key, value in iteritems(flat_dict):
        parts = key.split(sep)
        d = unflatted_dict
        for part in parts[:-1]:
            if part not in d:
                d[part] = {}
            d = d[part]
        d[parts[-1]] = value
    return unflatted_dict


def unique_preserve_order(iterable):
    if not iterable:
        return iterable
    return list(collections.OrderedDict.fromkeys(iterable).keys())


def unique_unhashable(collection):
    uniq = []
    for item in collection:
        if item not in uniq:
            uniq.append(item)
    return uniq


def random_ipv4():
    # Генерим только адреса, которые не используются пользователями, чтобы не
    # перегреть кому-нибудь счётчики в проде.
    # 10.0.0.0/8
    return str(netaddr.IPAddress((10 << 24) | random.getrandbits(24)))


def random_ipv6():
    # Генерим только адреса, которые не используются пользователями, чтобы не
    # перегреть кому-нибудь счётчики в проде.
    # 2001:db8::/32
    return str(netaddr.IPAddress((0x20010db8 << 96) | random.getrandbits(96)))


class MixinDependencyChecker(type):
    """
    Помогает найти все необходимые для миксина зависимости

    class Amixin(object):
        __metaclass__ = MixinDependencyChecker

        def a(self):
            return 'a'


    class Bmixin(object):
        __metaclass__ = MixinDependencyChecker

        mixin_dependencies = [Amixin]

        def b(self):
            return self.a() + 'b'


    class Foo(Bmixin):
        pass

    Здесь создали класс Foo, который использует миксин Bmixin. Но забыли
    подключить Amixin, который нужен для работы Bmixin.

    ValueError: Class Amixin is not superclass of Foo which is required by Bmixin
    """

    def __new__(cls, name, bases, attrs):
        kls = type.__new__(cls, name, bases, attrs)
        MixinDependencyChecker.check_mixin_deps(kls, bases)
        return kls

    @staticmethod
    def check_mixin_deps(cls, mixins):
        get_mixin_deps = lambda m: getattr(m, 'mixin_dependencies', list())

        for mixin in mixins:
            for dep in get_mixin_deps(mixin):
                if not issubclass(cls, dep):
                    raise ValueError(
                        'Class %s is not superclass of %s which is required by %s' %
                        (dep.__name__, cls.__name__, mixin.__name__),
                    )


def universal_decorator(outer):
    """
    Такой декоратор может быть вызван как
    decorator
    или
    decorator(some_args...)

    Все аргументы декоратора должны иметь значения по умолчанию
    """
    def inner(*args, **kwargs):
        if len(args) == 1 and not kwargs and callable(args[0]):
            return outer()(args[0])
        else:
            return outer(*args, **kwargs)

    return inner


def int_to_bytes(number, length, byteorder):
    """
    Преобразует неотрицательное целое число в цепочку байтов
    """
    assert isinstance(number, six.integer_types), 'Number should be integer: %s' % number
    assert number >= 0, 'Number should be equal or greater than 0: %s' % number
    assert byteorder in {'big', 'little'}, 'Unknown byteorder: %s' % byteorder

    required_bytes, remainder = divmod(type(number).bit_length(number), 8)
    if remainder:
        required_bytes += 1
    if length < required_bytes:
        raise ValueError('Length should be at least %d' % required_bytes)

    octets = list()
    for _ in range(length):
        octets.insert(0, number & 255)
        number >>= 8

    if byteorder == 'little':
        octets.reverse()

    _bytes = ''.join(map(chr, octets))
    if isinstance(_bytes, six.text_type):
        _bytes = _bytes.encode('latin1')
    return _bytes
