import errno
from grp import (
    getgrgid,
    getgrnam,
)
import os
from os import path
from pwd import (
    getpwnam,
    getpwuid,
)

import six


try:
    # Модуль passport.utils используется во внешних pypi-модулях
    # Аркадийно-специфическое заворачиваем в try

    from library.python import resource
    _resources = list(resource.iterkeys())

    def _get_resources():
        return _resources

except ImportError:
    def _get_resources():
        raise NotImplementedError()


def read_file(filepath, sources_list=None, encoding=None):
    if not sources_list:
        sources_list = ['file', 'binary']

    r = None

    for source in sources_list:
        if source == 'file' and os.path.exists(filepath):
            with open(filepath) as f:
                r = f.read()
                break
        elif source == 'binary':
            if filepath in _get_resources():
                r = resource.find(filepath)
                break
    if r is None:
        raise IOError("[Errno 2] No such file or directory: '%s'" % str(filepath))

    if encoding:
        return r.decode(encoding)
    return r


def list_resources():
    return list(_get_resources())


def path_exists(filepath, sources_list=None):
    if not sources_list:
        sources_list = ['file', 'binary']
    return (
        ('file' in sources_list and path.exists(filepath)) or
        ('binary' in sources_list and filepath in _get_resources())
    )


def mkdir_p(filepath):
    try:
        os.makedirs(filepath)
    except OSError as exc:
        if exc.errno == errno.EEXIST and os.path.isdir(filepath):
            pass
        else:
            raise


class chdir(object):
    def __init__(self, directory):
        self.directory = directory
        self.current_dir = os.getcwd()

    def __enter__(self):
        os.chdir(self.directory)

    def __exit__(self, exc_type, exc_val, exc_tb):
        os.chdir(self.current_dir)


def parse_owner(owner):
    owner = owner or ''
    owner = owner.strip()

    user = None
    group = None

    splitted = owner.split(':', 2)
    if len(splitted) > 0:
        user = splitted[0].strip() or None
        if len(splitted) > 1:
            group = splitted[1].strip() or None

    return user, group


def get_uid(name):
    """Returns an uid, given a user name or uid"""
    if getpwnam is None or name is None:
        return None

    result = None
    try:
        result = getpwnam(name)
    except KeyError:
        pass

    if result is None and str(name).isdigit():
        try:
            result = getpwuid(int(name))
        except KeyError:
            pass

    if result is not None:
        return result.pw_uid
    return None


def get_gid(name):
    """Returns a gid, given a group name or gid"""
    if getgrnam is None or name is None:
        return None

    result = None
    try:
        result = getgrnam(name)
    except KeyError:
        pass

    if result is None and str(name).isdigit():
        try:
            result = getgrgid(int(name))
        except KeyError:
            pass

    if result is not None:
        return result.gr_gid
    return None


def chown(path, user=None, group=None):
    """
    Backported from Python 3 (shutil)
    Change owner user and group of the given path.
    user and group can be the uid/gid or the user/group names, and in that case,
    they are converted to their respective uid/gid.
    """
    if user is None and group is None:
        raise ValueError("user and/or group must be set")

    _user = user
    _group = group

    # -1 means don't change it
    if user is None:
        _user = -1
    # user can either be an int (the uid) or a string (the system username)
    elif isinstance(user, six.string_types):
        _user = get_uid(user)
        if _user is None:
            raise LookupError("no such user: {!r}".format(user))

    if group is None:
        _group = -1
    elif not isinstance(group, int):
        _group = get_gid(group)
        if _group is None:
            raise LookupError("no such group: {!r}".format(group))

    os.chown(path, _user, _group)


def parent_chain(filepath):
    filepath = os.path.abspath(filepath)
    while filepath != '/':
        filepath = os.path.dirname(filepath)
        yield filepath


def access(filepath, source='file'):
    if source == 'file':
        return os.access(filepath, os.R_OK)
    else:
        return filepath in _get_resources()
