import contextlib
import errno
import fcntl
import os
import sys


class AlreadyLocked(Exception):
    pass


def makedirs(path):
    if not path:
        return

    try:
        if sys.version_info[0] == 2:
            os.makedirs(path)
        else:
            os.makedirs(path, exist_ok=True)
    except OSError:
        pass


@contextlib.contextmanager
def acquire(filename):
    filename = os.path.expanduser(filename)

    dir_path = os.path.dirname(filename)
    makedirs(dir_path)

    with open(filename, 'w') as f:
        try:
            fcntl.flock(f.fileno(), fcntl.LOCK_EX | fcntl.LOCK_NB)
        except IOError as e:
            if e.errno == errno.EAGAIN:
                raise AlreadyLocked(filename)
            else:
                raise

        yield

        fcntl.flock(f.fileno(), fcntl.LOCK_UN)
    try:
        os.remove(filename)
    except OSError:
        pass
