# -*- coding: utf-8 -*-

from __future__ import absolute_import

from cpython.mem cimport (
    PyMem_Free,
    PyMem_Malloc,
)
from cryptography import utils
import cryptography.exceptions
from cryptography.hazmat.backends.openssl.ec import (
    _check_signature_algorithm,
    _EllipticCurvePrivateKey,
    _EllipticCurvePublicKey,
    _ec_key_curve_sn,
    _mark_asn1_named_ec_curve,
)
from cryptography.hazmat.backends.openssl.utils import _calculate_digest_and_algorithm
from cryptography.hazmat.primitives import hashes
from cryptography.hazmat.primitives.asymmetric import ec
from passport.backend.utils.ccffi cimport ccffi


cdef extern from "openssl/ssl.h":
    ctypedef struct EC_KEY:
        pass

    ctypedef struct ENGINE:
        pass

    ctypedef struct EVP_PKEY:
        pass

    ctypedef struct EVP_PKEY_CTX:
        pass

    int EC_KEY_up_ref(EC_KEY* key)
    void EVP_PKEY_CTX_free(EVP_PKEY_CTX* ctx)
    EVP_PKEY_CTX* EVP_PKEY_CTX_new(EVP_PKEY* pkey, ENGINE* e)
    void* EVP_PKEY_get0(const EVP_PKEY* pkey)
    int EVP_PKEY_sign(
        EVP_PKEY_CTX* ctx,
        unsigned char* sig,
        size_t* siglen,
        const unsigned char* tbs,
        size_t tbslen,
    )
    int EVP_PKEY_sign_init(EVP_PKEY_CTX* ctx)
    int EVP_PKEY_size(const EVP_PKEY* pkey)
    int EVP_PKEY_verify(
        EVP_PKEY_CTX* ctx,
        const unsigned char* sig,
        size_t siglen,
        const unsigned char* tbs,
        size_t tbslen,
    )
    int EVP_PKEY_verify_init(EVP_PKEY_CTX* ctx)
    void OPENSSL_free(void* addr)
    void* OPENSSL_malloc(size_t num)


def GostEllipticCurve(**kwargs):
    cls = type('GostEllipticCurve', (object,), kwargs)
    cls = utils.register_interface(ec.EllipticCurve)(cls)
    return cls


def _EVP_PKEY_get1(evp_pkey):
    cdef EVP_PKEY* pkey = <EVP_PKEY*>ccffi.from_ffi_addr(evp_pkey)
    cdef EC_KEY* ec = <EC_KEY*>EVP_PKEY_get0(pkey)
    if ec != NULL:
        EC_KEY_up_ref(ec)
    return ccffi.to_ffi_addr(<void*>ec)


def _sn_to_elliptic_curve(backend, sn):
    try:
        return _CURVE_TYPES[sn]
    except KeyError:
        raise cryptography.exceptions.UnsupportedAlgorithm(
            "{} is not a supported elliptic curve".format(sn),
            cryptography.exceptions._Reasons.UNSUPPORTED_ELLIPTIC_CURVE,
        )


@utils.register_interface(hashes.HashAlgorithm)
class GostR3411_2012_256(
    # PKCS7SignatureBuilder разрешает подписывать только алгоритмами из
    # семейства SHA, поэтому приходиться наследоваться от соответствующего
    # класса
    hashes.SHA256,
):
    name = 'md_gost12_256'
    digest_size = 32
    block_size = 64


@utils.register_interface(hashes.HashAlgorithm)
class GostR3411_2012_512(
    # PKCS7SignatureBuilder разрешает подписывать только алгоритмами из
    # семейства SHA, поэтому приходиться наследоваться от соответствующего
    # класса
    hashes.SHA512,
):
    name = 'md_gost12_512'
    digest_size = 64
    block_size = 64


class _GostPrivateKey(_EllipticCurvePrivateKey):
    def __init__(self, backend, ec_key_cdata, evp_pkey):
        self._backend = backend
        self._ec_key = ec_key_cdata
        self._evp_pkey = evp_pkey

        sn = _ec_key_curve_sn(backend, ec_key_cdata)
        self._curve = _sn_to_elliptic_curve(backend, sn)
        _mark_asn1_named_ec_curve(backend, ec_key_cdata)

    def sign(self, data, signature_algorithm):
        _check_signature_algorithm(signature_algorithm)
        data, _ = _calculate_digest_and_algorithm(self._backend, data, signature_algorithm._algorithm)

        cdef:
            EVP_PKEY_CTX* ctx = NULL
            unsigned char* sig = NULL
            size_t siglen = 0

        try:
            siglen = EVP_PKEY_size(<const EVP_PKEY*>ccffi.from_ffi_addr(self._evp_pkey))
            sig = <unsigned char*>PyMem_Malloc(siglen)
            if sig == NULL:
                raise MemoryError()

            ctx = EVP_PKEY_CTX_new(<EVP_PKEY*>ccffi.from_ffi_addr(self._evp_pkey), NULL)
            self._backend.openssl_assert(EVP_PKEY_sign_init(ctx) == 1)
            self._backend.openssl_assert(EVP_PKEY_sign(ctx, sig, &siglen, data, len(data)) == 1)

            py_sig = sig[:siglen]
            return py_sig
        finally:
            if ctx != NULL:
                EVP_PKEY_CTX_free(ctx)
                ctx = NULL
            if sig != NULL:
                 PyMem_Free(sig)
                 sig = NULL


class _GostPublicKey(_EllipticCurvePublicKey):
    def __init__(self, backend, ec_key_cdata, evp_pkey):
        self._backend = backend
        self._ec_key = ec_key_cdata
        self._evp_pkey = evp_pkey

        sn = _ec_key_curve_sn(backend, ec_key_cdata)
        self._curve = _sn_to_elliptic_curve(backend, sn)
        _mark_asn1_named_ec_curve(backend, ec_key_cdata)

    def verify(self, signature, data, signature_algorithm):
        _check_signature_algorithm(signature_algorithm)
        data, _ = _calculate_digest_and_algorithm(self._backend, data, signature_algorithm._algorithm)

        cdef EVP_PKEY_CTX* ctx = EVP_PKEY_CTX_new(<EVP_PKEY*>ccffi.from_ffi_addr(self._evp_pkey), NULL)
        try:
            self._backend.openssl_assert(EVP_PKEY_verify_init(ctx) == 1)

            res = EVP_PKEY_verify(
                ctx,
                signature,
                len(signature),
                data,
                len(data),
            )
            if res != 1:
                self._backend._consume_errors()
                raise cryptography.exceptions.InvalidSignature
        finally:
            if ctx != NULL:
                EVP_PKEY_CTX_free(ctx)


_CURVE_TYPES = {c.name: c for c in [
    GostEllipticCurve(
        name='id-GostR3410-2001-CryptoPro-A-ParamSet',
        key_size=256,
    ),
    GostEllipticCurve(
        name='id-GostR3410-2001-CryptoPro-B-ParamSet',
        key_size=256,
    ),
    GostEllipticCurve(
        name='id-GostR3410-2001-CryptoPro-C-ParamSet',
        key_size=256,
    ),
    GostEllipticCurve(
        name='id-GostR3410-2001-CryptoPro-XchA-ParamSet',
        key_size=256,
    ),
    GostEllipticCurve(
        name='id-GostR3410-2001-CryptoPro-XchB-ParamSet',
        key_size=256,
    ),
    GostEllipticCurve(
        name='id-GostR3410-2001-TestParamSet',
        key_size=256,
    ),
    GostEllipticCurve(
        name='id-tc26-gost-3410-2012-256-paramSetA',
        key_size=256,
    ),
    GostEllipticCurve(
        name='id-tc26-gost-3410-2012-256-paramSetB',
        key_size=256,
    ),
    GostEllipticCurve(
        name='id-tc26-gost-3410-2012-256-paramSetC',
        key_size=256,
    ),
    GostEllipticCurve(
        name='id-tc26-gost-3410-2012-256-paramSetD',
        key_size=256,
    ),
    GostEllipticCurve(
        name='id-tc26-gost-3410-2012-512-paramSetA',
        key_size=512,
    ),
    GostEllipticCurve(
        name='id-tc26-gost-3410-2012-512-paramSetB',
        key_size=512,
    ),
    GostEllipticCurve(
        name='id-tc26-gost-3410-2012-512-paramSetC',
        key_size=512,
    ),
    GostEllipticCurve(
        name='id-tc26-gost-3410-2012-512-paramSetTest',
        key_size=512,
    ),
]}
