# -*- coding: utf-8 -*-

from __future__ import absolute_import

from cryptography import utils
from cryptography.hazmat.primitives.asymmetric import ec
from passport.backend.utils import gost

from .ccryptography import (
    _EVP_PKEY_get1,
    _GostPrivateKey,
    _GostPublicKey,
)


class GostBackendMixin(object):
    def __init__(self, *args, **kwargs):
        gost.install_gost_to_openssl()
        super(GostBackendMixin, self).__init__(*args, **kwargs)

    def _evp_pkey_to_private_key(self, evp_pkey):
        return self._evp_pkey_to_key(evp_pkey, 'private')

    def _evp_pkey_to_public_key(self, evp_pkey):
        return self._evp_pkey_to_key(evp_pkey, 'public')

    def _evp_pkey_to_key(self, evp_pkey, _key_type):
        if _key_type == 'private':
            Key = GostPrivateKey
            fallback = super(GostBackendMixin, self)._evp_pkey_to_private_key
        elif _key_type == 'public':
            Key = GostPublicKey
            fallback = super(GostBackendMixin, self)._evp_pkey_to_public_key
        else:
            raise NotImplementedError(_key_type)

        key_type = self._lib.EVP_PKEY_id(evp_pkey)

        if key_type in [gost.NID_id_GostR3410_2012_256, gost.NID_id_GostR3410_2012_512]:
            ec_cdata = _EVP_PKEY_get1(evp_pkey)
            self.openssl_assert(ec_cdata != self._ffi.NULL)
            ec_cdata = self._ffi.gc(ec_cdata, self._lib.EC_KEY_free)
            return Key(self, ec_cdata, evp_pkey)
        else:
            return fallback(evp_pkey)


@utils.register_interface(ec.EllipticCurvePrivateKeyWithSerialization)
class GostPrivateKey(_GostPrivateKey):
    def sign(self, data, signature_algorithm):
        return super(GostPrivateKey, self).sign(data, signature_algorithm)


@utils.register_interface(ec.EllipticCurvePublicKeyWithSerialization)
class GostPublicKey(_GostPublicKey):
    def verify(self, signature, data, signature_algorithm):
        super(GostPublicKey, self).verify(signature, data, signature_algorithm)
