# -*- coding: utf-8 -*-

import six


class BufferedReader(six.Iterator):
    def __init__(self, input_stream, filename=None):
        self.input_stream = input_stream
        self.filename = None
        self.buffer = []
        self.buffer_length = 0
        self.position = 0
        self.closed = False
        self.iter_chunk_length = 2 ** 20

    def __iter__(self):
        return self

    def __next__(self):
        data = self.read(size=self.iter_chunk_length)
        if not data:
            raise StopIteration()
        return data

    def tell(self):
        return self.position

    def flush(self):
        pass

    def fileno(self):
        raise OSError('Stream doesn\'t use fileno')

    def close(self):
        self.closed = True

    def closed(self):
        return self.closed

    def read_from_stream(self, size):
        raise NotImplementedError('Override me')  # pragma: no cover

    def read(self, size=-1):
        if size < -1:
            raise ValueError('bytes must be >= -1')
        if size == -1:
            retval = b''.join(self.input_stream)
            self.position = len(retval)
            return retval

        while self.buffer_length < size:
            data = self.read_from_stream(size)
            if not data:
                break
            self.buffer.append(data)
            self.buffer_length += len(data)

        if self.buffer_length > size:
            extra_bytes_to_return = self.buffer_length - size
            retval = b''.join(self.buffer)
            self.buffer_length = extra_bytes_to_return
            self.buffer = [retval[-extra_bytes_to_return:]]
            retval = retval[:-extra_bytes_to_return]
        else:
            retval = b''.join(self.buffer)
            self.buffer_length = 0
            self.buffer = []
        self.position += len(retval)
        return retval


class StrictSizeReader(BufferedReader):
    """
    Полезно, если низлежащий поток данных может отдавать больше байт, чем запрошено.
    Пример: Response от urllib3, который читает из сокета size байт, но затем декодирует поток
    и может отдавать больше байт, чем запрошено.
    """
    def read_from_stream(self, size):
        return self.input_stream.read(size)


class LazyFileLikeObject(BufferedReader):
    def read_from_stream(self, size):
        try:
            return next(self.input_stream)
        except StopIteration:
            # EOF
            return b''
