# coding: utf-8

import contextlib
import logging

import mock
import ylock
from ylock.backends.yt import YTTimeout
from ylock.backends.zookeeper import BackendConnectionError as ZookeeperConnectionError


def get_lock_manager(ylock_config):
    return ylock.create_manager(**ylock_config)


class LockError(Exception):
    pass


@contextlib.contextmanager
def lock(ylock_config, lock_name, skip_lock=False, block=False, timeout=None, logger=None, log_prefix=None):
    """
    Менеджер контекста пробует захватить блокировку и вызывает тело.
    Возвращает None, если блокировку не запрашивали, True — блокировка получена, False — не получена.
    """
    if logger is None:
        logger = logging.getLogger('lock')

    log_prefix = log_prefix or ''

    if skip_lock:
        logger.info('{}Skip lock'.format(log_prefix))
        yield None
        return

    lock_manager = get_lock_manager(ylock_config)

    backend = ylock_config['backend']
    hosts = lock_manager.hosts

    try:
        logger.info('{}Acquire lock "{}" ("{}", "{}")'.format(log_prefix, lock_name, backend, repr(hosts)))

        with lock_manager.lock(lock_name, block=block, timeout=timeout) as is_locked_successfully:
            if is_locked_successfully:
                logger.info('{}Lock acquired "{}" ("{}", "{}")'.format(log_prefix, lock_name, backend, repr(hosts)))
                yield True
                logger.info('{}Release lock "{}" ("{}", "{}")'.format(log_prefix, lock_name, backend, repr(hosts)))
            else:
                logger.info('{}Lock is not acquired "{}" ("{}", "{}")'.format(log_prefix, lock_name, backend, repr(hosts)))
                yield False
    except (ZookeeperConnectionError, YTTimeout) as e:
        logger.warning('{}Failed to connect to lock backend "{}" ({} "{}")'.format(log_prefix, backend, e.__class__.__name__, str(e)))
        raise LockError(e)
    finally:
        lock_manager.close()


@contextlib.contextmanager
def lock_mock():
    lock_manager = ylock.create_manager('thread')
    with mock.patch(
        'passport.backend.utils.lock.get_lock_manager',
        return_value=lock_manager,
    ):
        yield lock_manager
    lock_manager.close()
