# -*- coding: utf-8 -*-
from six.moves import zip_longest


YANDEXUID_TIMESTAMP_LENGTH = 10


def int_missing(missing=-1):
    def wrapper(value):
        return int(value) if value is not None else missing

    return wrapper


def float_missing(value, missing=-1, round_digits=4):
    return round(float(value), round_digits) if value is not None else missing


def probability(value, round_digits=4):
    return float_missing(value, missing=0, round_digits=round_digits)


def sparse(value):
    if isinstance(value, list):
        return ','.join(map(str, value))
    else:
        return value


def parse_uatraits_value(value):
    if not value or value == 'Unknown':
        return None
    return value


def parse_uatraits_bool_missing(value):
    if value is None or value == 'Unknown':
        return None
    return bool(value)


class Version(object):
    @staticmethod
    def parse(version):
        if not version:
            return None

        try:
            version_parts = map(int, version.split('.'))
        except ValueError:
            return None

        return Version(version_parts)

    def __init__(self, version_parts):
        self.version_parts = version_parts

    def __cmp__(self, other):
        if not isinstance(other, Version):
            raise ValueError('cannot compare Version with %s' % type(other))

        for version_part1, version_part2 in zip_longest(self.version_parts, other.version_parts):
            if version_part1 == version_part2:
                continue
            elif (version_part1 is not None and
                  (version_part2 is None or (version_part2 is not None and version_part1 > version_part2))):
                return 1
            else:
                return -1
        return 0


def parse_browser(browser):
    browser_parts = browser.split(' ')
    if len(browser_parts) == 2:
        browser_name = browser_parts[0]
        browser_version = Version.parse(browser_parts[1])
        return browser_name, browser_version
    elif len(browser_parts) == 1:
        return browser_parts[0], None
    else:
        return None, None


def parse_yandexuid_timestamp(cookie):
    if not cookie:
        return None

    timestamp = str(cookie)[-YANDEXUID_TIMESTAMP_LENGTH:]
    if not timestamp.isdigit():
        return None
    return int(timestamp)
