# -*- coding: utf-8 -*-
from __future__ import absolute_import

import codecs
import datetime
from decimal import Decimal
import string

import six


try:
    from html.entities import codepoint2name
except ImportError:
    from htmlentitydefs import codepoint2name


# Получаем все коды символов, которые могут быть заменены на HTML-entities
# вроде &amp;
_codepoints = list(codepoint2name.keys())
_codepoints.extend([
    ord('\''),  # + апостроф
    ord('\x00'),  # + нулевой символ
])
CODEPOINTS_WITH_HTML_ENTITY = frozenset(_codepoints)


# Функция взята из django
def is_protected_type(obj):
    """Determine if the object instance is of a protected type.

    Objects of protected types are preserved as-is when passed to
    smart_unicode(strings_only=True).
    """
    return isinstance(
        obj,
        (
            type(None),
            datetime.datetime,
            datetime.date,
            datetime.time,
            float,
            Decimal,
        ) + six.integer_types,
    )


# Взято из django 3.0
def smart_unicode_py3(s, encoding='utf-8', strings_only=False, errors='strict'):
    """
    Similar to smart_str(), except that lazy instances are resolved to
    strings, rather than kept as lazy objects.

    If strings_only is True, don't convert (some) non-string-like objects.
    """
    # Handle the common case first for performance reasons.
    if issubclass(type(s), str):
        return s
    if strings_only and is_protected_type(s):
        return s
    if isinstance(s, bytes):
        s = str(s, encoding, errors)
    else:
        s = str(s)
    return s


# Взято из django 3.0
def smart_bytes(s, encoding='utf-8', strings_only=False, errors='strict'):
    """
    Возвращаем набор байтов, представляющий собой закодированную в указанной
    кодировке строку.

    Если указан флаг strings_only, то не пытаемся закодировать некоторые
    объекты, которые не являются строками.
    """
    if strings_only and isinstance(s, (type(None), int)):
        return s
    elif isinstance(s, six.binary_type):
        if encoding != 'utf-8':
            return s.decode('utf-8', errors).encode(encoding, errors)
        return s
    elif isinstance(s, six.text_type):
        return s.encode(encoding, errors)
    else:
        try:
            s = str(s)
            if isinstance(s, six.text_type):
                return s.encode(encoding, errors)
            return s
        except UnicodeEncodeError:
            if isinstance(s, Exception):
                # Случай с подклассом Exception, в котором содержатся данные
                # не в ASCII и который не знает как вывести себя на экран
                # правильно. В этом случае не нужно перекидывать исключение.
                return b' '.join([
                    smart_str(arg, encoding, strings_only, errors) for arg in s
                ])
            return six.text_type(s).encode(encoding, errors)


def always_str(s, encoding='utf-8', errors='strict'):
    """
    В любом python вернёт str. Сделает умный decode/encode для unicode/bytes.
    Можно использовать, когда встречается библиотека, которая в python 2 и 3
    принимает str, что встречается нередко

    Если указан флаг strings_only, то не пытаемся закодировать некоторые
    объекты, которые не являются строками.
    """
    if isinstance(s, six.binary_type):
        if six.PY3:
            return s.decode(encoding, errors)
        else:
            return s
    elif isinstance(s, six.text_type):
        if six.PY2:
            return s.encode(encoding, errors)
        else:
            return s
    else:
        return str(s)


def compare_strings(str_a, str_b):
    """Сравнивает две строки.

    Время работы зависит от длины строк и не зависит от
    расстояния между началом строки и первым несовпадающим
    символом.

    """

    if len(str_a) != len(str_b):
        diff = 1
        str_a = str_b
    else:
        diff = 0
    for char_a, char_b in zip(str_a, str_b):
        diff |= ord(char_a) ^ ord(char_b)

    return diff == 0


# Функция взята из django
def smart_unicode_py2(s, encoding='utf-8', strings_only=False, errors='strict'):
    """
    Returns a unicode object representing 's'. Treats bytestrings using the
    'encoding' codec.

    If strings_only is True, don't convert (some) non-string-like objects.
    """
    # Handle the common case first, saves 30-40% in performance when s
    # is an instance of unicode. This function gets called often in that
    # setting.
    if isinstance(s, six.text_type):
        return s
    if strings_only and is_protected_type(s):
        return s
    try:
        if not isinstance(s, six.string_types,):
            if hasattr(s, '__unicode__'):
                s = six.text_type(s)
            else:
                try:
                    # уже unicode
                    if isinstance(s, six.text_type):
                        pass
                    # строка или байты, которые надо превратить в unicode
                    elif isinstance(s, six.binary_type):
                        s = s.decode(encoding, errors)
                    # всё остальное - repr + unicode
                    else:
                        try:
                            s = str(s)
                        except TypeError:
                            s = repr(s)
                        s = six.text_type(s)
                except UnicodeEncodeError:
                    if not isinstance(s, Exception):
                        raise
                    # If we get to here, the caller has passed in an Exception
                    # subclass populated with non-ASCII data without special
                    # handling to display as a string. We need to handle this
                    # without raising a further exception. We do an
                    # approximation to what the Exception's standard str()
                    # output should be.
                    s = u' '.join([
                        smart_unicode(arg, encoding, strings_only,
                                      errors) for arg in s
                    ])
        elif not isinstance(s, six.text_type):
            # Note: We use .decode() here, instead of unicode(s, encoding,
            # errors), so that if s is a SafeString, it ends up being a
            # SafeUnicode at the end.
            s = s.decode(encoding, errors)
    except UnicodeDecodeError:
        if not isinstance(s, Exception):
            raise
        else:
            # If we get to here, the caller has passed in an Exception
            # subclass populated with non-ASCII bytestring data without a
            # working unicode method. Try to handle this without raising a
            # further exception by individually forcing the exception args
            # to unicode.
            s = u' '.join([
                smart_unicode(arg, encoding, strings_only,
                              errors) for arg in s
            ])
    return s


def mask_prefix(string_value, unmask_length):
    """
    Заменить звёздочкой все символы в строке, кроме последних unmask_length.

    Например, mask_prefix('Андрей', 2) == '****ей'.
    """
    if unmask_length < 0:
        raise ValueError(u'unmask_length should be greater of equal to zero')
    length = len(string_value)
    if length <= unmask_length:
        return string_value
    star_count = length - unmask_length
    return '*' * star_count + string_value[star_count:]


def mask_postfix(string_value, mask_length):
    """
    Заменить звёздочкой последние mask_length литер в строке.

    Например, mask_postfix('Андрей', 2) == 'Андр**'.
    """
    if mask_length < 0:
        raise ValueError(u'mask_length should be greater of equal to zero')
    length = len(string_value)
    if length <= mask_length:
        return '*' * length
    return string_value[:length - mask_length] + '*' * mask_length


def _to_json_unicode_entity(char_code):
    """
    Подготавливает JSON-представление unicode-символа по его коду

    >>> _to_json_unicode_entity(0)
    '\\u0000'
    >>> _to_json_unicode_entity(62)
    '\\u003e'
    """
    return r'\u%04x' % char_code


def _to_unicode_entity_if_needed(char):
    """
    >>> _to_unicode_entity_if_needed('&')
    '\\u0026'
    >>> _to_unicode_entity_if_needed('a')
    'a'
    """
    char_code = ord(char)
    if char_code in CODEPOINTS_WITH_HTML_ENTITY:
        return _to_json_unicode_entity(char_code)

    return char


def escape_special_chars_to_unicode_entities(text):
    """
    Выполняет замену "опасных" символов вроде <>'& в строке на их
    unicode-представления для JSON.
    """
    return ''.join(
        map(
            _to_unicode_entity_if_needed,
            text,
        ),
    )


PRINTABLE_ASCII_CHARS = string.ascii_letters + string.digits + string.punctuation + ' '


def escape_unprintable_bytes(s):
    if six.PY3:
        s = smart_text(s, encoding='ascii', errors='backslashreplace')
    return ''.join(c if c in PRINTABLE_ASCII_CHARS else r'\x{0:02x}'.format(ord(c)) for c in s)


def snake_case_to_camel_case(s, first_lower=False):
    splitted = s.split('_')
    return ''.join(map(
        lambda x: x[1] if first_lower and x[0] == 0 else x[1].capitalize(),
        enumerate(splitted),
    ))


def encoding_aware_split_bytes(
    _bytes,
    byte_length,
    incremental_decoder=None,
    encoding='utf-8',
    align_byte=None,
):
    """
    Разбивает строку на строки длины не более byte_length байтов, учитывая
    кодировку, так чтобы байты составляющие литеру не разбивались по разным
    строкам.

    Возвращает юникодные строки
    """
    if not incremental_decoder:
        incremental_decoder = codecs.getincrementaldecoder(encoding)()

    align_char = align_byte[0].decode(encoding) if align_byte else None

    bytes_to_read = byte_length
    tail = _bytes
    lines = list()
    while tail:
        head, tail = tail[:bytes_to_read], tail[bytes_to_read:]
        line = incremental_decoder.decode(head, final=not tail)

        bytes_to_read = byte_length - len(incremental_decoder.getstate()[0])

        if align_char and tail:
            # Дополняем все строки, кроме последней до максимальной длины, таким
            # образом начала строк будут выравнены по границе byte_length

            # Пользуемся тем, что len(incremental_decoder.getstate()[0]) =
            # числу байт, которыми нужно дополнить строку line, чтобы её длина
            # стала byte_length
            line += align_char * len(incremental_decoder.getstate()[0])

        lines.append(line)

    return lines


# NOTE В джанго smart_str на py3 ведёт себя как smart_text, у нас не так
smart_str = smart_bytes

if six.PY2:
    smart_unicode = smart_unicode_py2
    smart_text = smart_unicode_py2
else:
    smart_unicode = smart_unicode_py3
    smart_text = smart_unicode_py3
