# -*- coding: utf-8 -*-

from __future__ import absolute_import

from datetime import (
    date,
    datetime,
    timedelta,
)
from math import floor
import time

import pytz
import six


DATE_FORMAT = '%Y-%m-%d'
DEFAULT_FORMAT = '%Y-%m-%d %H:%M:%S'
DATE_WITH_TZ_FORMAT = '%Y-%m-%d %Z%z'


def datetime_to_unixtime(dt):
    """Converting from datetime type to unixtime
    @dt datetime
    @return long
    """
    try:
        return time.mktime(dt.timetuple()) + 1e-6 * dt.microsecond
    except AttributeError:
        raise TypeError("datetime_to_unixtime expects datetime object, got %s instead" % type(dt))


def datetime_to_integer_unixtime(dt):
    return int(round_timestamp(datetime_to_unixtime(dt)))


def datetime_to_integer_unixtime_nullable(dt):
    return datetime_to_integer_unixtime(dt) if dt else None


def unixtime_to_datetime(unixtime):
    """Converts unixtime(int or string with int) to dateime
    :param unixtime: unix timestamp (int or string)
    :return datetime object
    """
    try:
        return datetime.fromtimestamp(int(unixtime))
    except ValueError:
        raise TypeError('unixtime_to_datetime expects int object, got %s instead' % type(unixtime))


def parse_datetime(value, format=DEFAULT_FORMAT, date_only=False):
    if not value:
        return

    if isinstance(value, datetime):
        return value
    elif value == '0000-00-00 00:00:00':
        result = datetime.fromtimestamp(0)
    else:
        result = datetime.strptime(value, format)

    if date_only:
        return date(*result.timetuple()[:3])
    else:
        return result


def parse_unixtime(value):
    if not value:
        return

    return unixtime_to_datetime(value)


def string_to_integer_unixtime(value):
    return int(round_timestamp(float(value)))


def datetime_to_string(dt, format=DEFAULT_FORMAT):
    return dt.strftime(format)


def timeuuid_to_timestamp(timeuuid):
    """
    Вытаскивает timestamp из timeuuid. Подробности см. в uuid.uuid1
    """
    return int(round_timestamp((timeuuid.time - 0x01b21dd213814000) * 100 / 1e9))


def switch_datetime_tz(aware_datetime, tz):
    """
    Правильно переключить часовой пояс для datetime-объекта со знанием о
    часовом поясе.
    """
    return tz.normalize(aware_datetime.astimezone(tz))


def safe_local_datetime_to_date(dt, user_tz):
    """
    Преобразовать datetime в дату, с учетом часового пояса пользователя.
    Позволяет учитывать ситуации со сдвигом +- 1 день из-за различных часовых поясов.
    @param dt: datetime объект без знания о часовом поясе (unaware), время в поясе Europe/Moscow.
    @param user_tz: часовой пояс пользователя.
    @return unaware-datetime объект, содержащий только дату, в часовом поясе user_tz.
    """
    local_tz = pytz.timezone('Europe/Moscow')
    zero_time = dict(hour=0, minute=0, second=0, microsecond=0)
    if user_tz != local_tz:
        # Перевести в часовой пояс пользователя, отбросить время - таким образом
        # получим дату в часовом поясе пользователя.
        return switch_datetime_tz(
            local_tz.localize(dt),
            user_tz,
        ).replace(tzinfo=None, **zero_time)
    return dt.replace(**zero_time)


def get_unixtime():
    return int(round_timestamp(time.time()))


def round_timestamp(timestamp):
    if isinstance(timestamp, datetime):
        timestamp = timestamp.replace(microsecond=0)
    elif isinstance(timestamp, float):
        timestamp = floor(timestamp)
    return timestamp


class _ZeroDatetime(datetime):
    """
    Имитация объекта datetime(0, 0, 0, 0, 0).
    Для работы с БД и тестов.
    """
    def __new__(cls):
        earliest = datetime.min
        return super(_ZeroDatetime, cls).__new__(
            cls,
            earliest.year,
            earliest.month,
            earliest.day,
        )

    def __str__(self):
        return '0000-00-00 00:00:00'

    def __repr__(self):
        return 'datetime.datetime(0, 0, 0, 0, 0)'

    def replace(self, *args, **kwargs):
        return self

    def __eq__(self, other):
        if not isinstance(other, datetime):
            return False
        return (
            self.year == other.year and
            self.month == other.month and
            self.day == other.day and
            self.hour == other.hour and
            self.minute == other.minute and
            self.second == other.second and
            self.microsecond == other.microsecond
        )

    def __ne__(self, other):
        return not self.__eq__(other)

    year = month = day = hour = minute = second = microsecond = property(lambda self: 0)


zero_datetime = _ZeroDatetime()


class Weekday(object):
    MONDAY = 1
    TUESDAY = 2
    WEDNESDAY = 3
    THURSDAY = 4
    FRIDAY = 5
    SATURDAY = 6
    SUNDAY = 7


class Period(object):
    SECOND = 1
    MINUTE = 60 * SECOND
    HOUR = 60 * MINUTE
    QUARTER = 4 * HOUR
    DAY = 24 * HOUR
    WEEK = 7 * DAY
    MONTH = 30 * DAY
    YEAR = 365 * DAY


def to_date_str(date):
    return date.strftime(DATE_FORMAT)


def from_str_to_date(date):
    return datetime.strptime(date, DATE_FORMAT)


class Daterange(six.Iterator):
    def __init__(self, date_start=None, date_end=None, outer_mapper=None):
        if date_start and not date_end:
            self.date_start = date_start
            self.date_end = date_start
        elif not date_start and not date_end:
            self.date_start = date.today()
            self.date_end = self.date_start
        elif date_start and date_end:
            self.date_start = date_start
            self.date_end = date_end
        else:
            raise ValueError('date_end cannot be the only set argument')
        self.current = self.date_start - timedelta(days=1)
        self.output_mapper = outer_mapper
        if not self.output_mapper:
            self.output_mapper = lambda x: x

    def __iter__(self):
        return self

    def __next__(self):
        self.current += timedelta(days=1)
        if self.current <= self.date_end:
            return self.output_mapper(self.current)
        else:
            raise StopIteration

    def reset(self):
        self.current = self.date_start - timedelta(days=1)

    def __len__(self):
        return (self.date_end - self.date_start).days

    def as_unixtime(self):
        new = Daterange(self.date_start, self.date_end, outer_mapper=lambda x: int(x.strftime("%s")))
        return new

    @staticmethod
    def from_str(string):
        dates = list(map(from_str_to_date, string.lstrip('{').rstrip('}').split('..')))
        return Daterange(*dates)


daterange = Daterange
