# -*- coding: utf-8 -*-
from contextlib import contextmanager
from datetime import datetime
import logging

from passport.backend.utils.string import smart_str
from passport.backend.utils.time import datetime_to_string
import six


MAX_ENTRY_LENGTH = 10000

MASK_PLACEHOLDER = '*****'

REPLACE_MAP = [
    ('\\', r'\\'),
    ('\t', r'\t'),
    ('\n', r'\n'),
    ('\r', r'\r'),
    ('\0', r'\0'),
]


def escape_text_for_log(text, errors='strict'):
    text = smart_str(text, errors=errors)
    for char, replacement in REPLACE_MAP:
        text = text.replace(char, replacement)
    return text


class TskvLogEntry(object):
    """Представляет переданные параметры в форме tskv"""

    def __init__(self, **params):
        self.params = params

    @staticmethod
    def escape_value(value):
        return escape_text_for_log(value)

    @staticmethod
    def truncate_str(text):
        return text[:MAX_ENTRY_LENGTH]

    def postprocess_key_and_value(self, key, value):
        return key, value

    def __eq__(self, other):
        if not isinstance(other, TskvLogEntry):
            return False
        return self.params == other.params

    def __ne__(self, other):
        return not (self == other)

    def __unicode__(self):
        return str(self).decode('utf-8')

    def __str__(self):
        lines = ['tskv']
        for key, value in six.iteritems(self.params):
            if value is not None:
                key = smart_str(key)
                if isinstance(value, datetime):
                    # Подразумеваем что дата и время приводятся в правильный формат
                    value = self.datetime_to_str(value)
                elif isinstance(value, bool):
                    value = tskv_bool(value)
                else:
                    value = self.truncate_str(
                        self.escape_value(value)
                    )
                key, value = self.postprocess_key_and_value(key, value)
                line = '%s=%s' % (key, value)
                lines.append(line)
        return '\t'.join(lines)

    @staticmethod
    def datetime_to_str(value):
        return datetime_to_string(value)


class TskvLogger(object):
    """Основной класс для ведения журнала"""

    default_logger_name = None
    entry_class = None

    def __init__(self, python_logger=None, **context):
        self._logger = python_logger or self.get_default_logger()
        self._context = context.copy()
        self._current_values = {}
        self._stashes = []
        self._context_stack = []

    @classmethod
    def get_default_logger(cls):
        if cls.default_logger_name is None:
            raise NotImplementedError()  # pragma: no cover
        return logging.getLogger(cls.default_logger_name)

    def get_child(self, **context):
        """Возвращает новый логгер с контекстом, равным текущему контексту + context"""
        child_logger = self.__class__(**self._context)
        child_logger.bind_context(**context)
        return child_logger

    def _bind(self, storage, args, values):
        for arg in args:
            if not isinstance(arg, dict):
                raise TypeError('Can only process dicts')
            storage.update(self._unlink(arg))
        storage.update(self._unlink(values))

    def _unlink(self, new):
        retval = {}
        for key, value in new.items():
            if isinstance(value, self.Link) and key in self.all_values:
                value = u'.'.join([str(self.all_values[key]), str(value)])
            retval[key] = value
        return retval

    def bind(self, *args, **values):
        """Добавляет значения к текущим данным"""
        self._bind(self._current_values, args, values)

    def bind_context(self, *args, **context):
        """Добавляет значения к контексту логгера"""
        self._bind(self._context, args, context)

    @property
    def all_values(self):
        """Возвращает все значения, которые ожидают записи в лог"""
        all_values = self._context.copy()
        all_values.update(self._current_values)
        return all_values

    @property
    def has_data(self):
        return bool(self.all_values)

    def _write_to_log(self, data):
        self._logger.debug(data)

    def _build_entry(self):
        if self.entry_class is None:
            raise NotImplementedError()  # pragma: no cover
        return self.entry_class(**self.all_values)

    def log(self, **values):
        """Пишет в лог контекст + текущие данные + values, очищает текущие данные"""
        self.bind(**values)
        if self.has_data:
            self._write_to_log(str(self._build_entry()))
            self._current_values = {}

    @contextmanager
    def make_context(self, **context):
        """
        Этот контекстный менеджер может на время своей работы дополнить
        или переписать контекст журналиста.
        """
        self._context_stack.append(self._context.copy())
        self._context.update(self._unlink(context))
        try:
            yield
        finally:
            self._context = self._context_stack.pop()

    def stash(self, **values):
        """Сохраняет контекст + текущие данные + values в виде черновика записи, очищает текущие данные"""
        self.bind(**values)
        if self.has_data:
            self._stashes.append(self.all_values)
            self._current_values = {}

    @property
    def all_stashes(self):
        """Возвращает все черновики, сохранённые командой stash и ожидающие записи в лог"""
        return self._stashes

    def dump_stashes(self, **values):
        """Записывает в лог все черновики, сохранённые командой stash"""
        for stash in self.all_stashes:
            stash.update(values)
            self.log(**stash)
        self._stashes = []

    class Link(six.text_type):
        pass


def tskv_bool(value):
    return '1' if value else '0'
