#!/usr/bin/env python
# coding: utf-8

from gevent import monkey  # isort:skip
monkey.patch_all()  # isort:skip

import logging
import traceback

from flask import (
    Flask,
    jsonify,
    request,
)
from passport.backend.core.lazy_loader import LazyLoader
from passport.backend.utils import p3
from passport.backend.utils.common import noneless_dict
from passport.backend.vault.api.commands import create_cli
from passport.backend.vault.api.db import (
    configure_migrations,
    get_db,
)
from passport.backend.vault.api.errors import BaseError
from passport.backend.vault.api.routes import configure_routes
from passport.backend.vault.api.utils.json import ExploitableJSONEncoder
from passport.backend.vault.api.utils.secrets import token_hex
from werkzeug.exceptions import HTTPException
from werkzeug.routing import RequestRedirect


def get_request_id():
    return request.headers.get('X-Request-Id') or token_hex(16)


def create_app(config, *args, **kwargs):
    app = Flask(__name__, static_folder=config['application']['static_folder'])
    app.config.update(config.get_config_by_basename('flask.yaml'))
    app.json_encoder = ExploitableJSONEncoder
    app.url_map.strict_slashes = False

    get_db().init_app(app)

    configure_routes(app)
    configure_migrations(app, config)

    @app.before_request
    def persist_request_data():
        request.request_id = get_request_id()
        request.user_ip = request.headers.get(
            'X-Ya-Consumer-User-Ip',
            request.headers.get('X-Forwarded-For', request.remote_addr),
        )
        request.real_user_agent = request.headers.get('X-Ya-Client-User-Agent', request.user_agent.string)
        request.remote_ip = request.headers.get('X-Real-Ip', request.remote_addr)

    @app.after_request
    def process_after_request(response):
        response.headers['X-Request-Id'] = request.request_id
        return response

    @app.after_request
    def log_request(response):
        logging.getLogger('request_logger').info(
            '{status_code} {content_length}'.format(
                status_code=response.status_code,
                content_length=response.calculate_content_length() or '-',
            ),
        )
        return response

    @app.errorhandler(Exception)
    def handle_exception(error):
        get_db().session.rollback()

        exception_logger = logging.getLogger('exception_logger')
        statbox_logger = logging.getLogger('statbox')
        traceback_logger = logging.getLogger('traceback_logger')

        traceback_value = u'\n' + p3.u8(traceback.format_exc())

        def enrich_error_dict(error):
            result = dict(error)
            result.setdefault('hostname', config.get('hostname'))
            result.setdefault('environment', config.get('environment'))
            result.setdefault('api_request_id', request.request_id)
            return result

        level = 'ERROR'
        if isinstance(error, BaseError):
            level = error.level
            response = jsonify(enrich_error_dict(
                error.as_dict(),
            ))
            response.status_code = error.status_code
            log_info = {'action': 'error'}
            log_info.update(error.as_private_dict())
            statbox_logger.info(log_info)
        elif isinstance(error, RequestRedirect):
            json_codes = {
                301: 'moved_permanently',
                302: 'moved_temporary',
            }
            response = jsonify(enrich_error_dict({
                'status': 'error',
                'code': json_codes.get(error.code, 'moved_permanently'),
            }))
            response.status_code = error.code
            response.headers['Location'] = error.new_url
        elif isinstance(error, HTTPException):
            json_codes = {
                404: 'page_not_found',
                405: 'method_not_allowed',
            }
            response = jsonify(noneless_dict(enrich_error_dict({
                'status': 'error',
                'code': json_codes.get(error.code, 'http_server_error'),
                'message': error.description,
            })))
            if error.code in json_codes:
                level = 'WARNING'
            response.status_code = error.code
        else:
            response = jsonify(enrich_error_dict({
                'status': 'error',
                'code': 'internal_server_error',
                'message': 'Internal server error',
            }))
            response.status_code = 500

        error_str = p3.u8(str(error))

        if level == 'ERROR':
            exception_logger.error(error_str)
            traceback_logger.error(traceback_value)
        elif level == 'WARNING':
            exception_logger.warning(error_str)
            traceback_logger.warning(traceback_value)
        return response

    return app


def main():
    config = LazyLoader.get_instance('config')
    app = create_app(config)
    cli = create_cli(app, config)
    cli()
