# -*- coding: utf-8 -*-

import logging

from passport.backend.core.builders import abc as passport_core_abc
from passport.backend.core.builders.abc.exceptions import BaseABCError
from passport.backend.vault.api.builders.loggers import VaultGraphiteLogger
from passport.backend.vault.api.errors import ABCError
from passport.backend.vault.api.models import (
    AbcRole,
    AbcScope,
)
import six


logger = logging.getLogger('info_logger')


class ABC(passport_core_abc.ABC):
    page_limit = 2 ** 31 - 1

    page_size = 500  # Для сервисов и людей ставим большой лимит (max: 1000)
    tvm_apps_page_size = 500  # Для ресурсов отдельный лимит, чтобы регулировать отдельно

    def __init__(self, config, *args, **kwargs):
        super(ABC, self).__init__(
            reconnect_on_retries=True,
            graphite_logger=VaultGraphiteLogger(service='abc'),
            *args,
            **kwargs
        )
        self.config = config

    def get_all_departments(self):
        params = {
            'state__in': 'develop,supported,needinfo',
            'fields': 'id,name,slug',
            'ordering': 'id',
            'page_size': self.page_size,
        }
        results = {}
        try:
            rv = self._make_request_with_cursor_pagination(
                url_suffix='v4/services/',
                method='GET',
                params=params,
                oauth_token=self.config['abc']['oauth_token'],
                max_pages=self.page_limit,
            )
        except BaseABCError as ex:
            raise ABCError(ex.message)
        for department in rv:
            department_id = department.get('id')
            if department_id is not None:
                department_id = int(department_id)
                results[department_id] = {
                    'name': department.get('name', {}).get('ru'),
                    'slug': department.get('slug'),
                }
        return results

    def get_roles_and_scopes(self, services=None):
        params = {
            'fields': 'id,name,service,scope,code',
            'ordering': 'id',
            'page_size': self.page_size,
        }
        rv = self._make_request_with_cursor_pagination(
            url_suffix='v4/roles/',
            method='GET',
            params=params,
            oauth_token=self.config['abc']['oauth_token'],
            max_pages=self.page_limit,
        )
        result_scopes = dict(
            map(
                lambda x: (int(x['scope']['id']), x['scope']),
                filter(
                    lambda x: x['service'] is None and x['scope'] is not None,
                    rv,
                ),
            )
        )
        result_roles = dict(
            map(
                lambda x: (int(x['id']), x),
                filter(
                    lambda x: x['service'] is None,
                    rv,
                ),
            )
        )
        default_scopes = set(result_scopes)
        default_roles = set(result_roles)

        result_services_scopes = dict()
        result_services_roles = dict()
        if services is not None:
            result_services_scopes = dict(
                map(
                    lambda x: (x, set(default_scopes)),
                    services,
                ),
            )
            result_services_roles = dict(
                map(
                    lambda x: (x, set(default_roles)),
                    services,
                ),
            )

        for role in filter(lambda x: x['service'] is not None and x['scope'] is not None, rv):
            service_id = int(role['service']['id'])
            scope_id = int(role['scope']['id'])
            service_scopes = result_services_scopes.setdefault(
                service_id,
                set(default_scopes),
            )
            if scope_id not in result_scopes:
                result_scopes[scope_id] = role['scope']
            if scope_id not in service_scopes:
                service_scopes.add(scope_id)

        for role in filter(lambda x: x['service'] is not None, rv):
            service_id = int(role['service']['id'])
            role_id = int(role['id'])
            services_roles = result_services_roles.setdefault(
                service_id,
                set(default_roles),
            )
            if role_id not in result_roles:
                result_roles[role_id] = role
            if role_id not in services_roles:
                services_roles.add(role_id)

        result_scopes = dict(map(
            lambda x: (x[0], AbcScope(
                id=x[0],
                unique_name=x[1]['slug'],
                display_name=x[1].get('name', {}).get('ru'),
            )),
            six.viewitems(result_scopes),
        ))

        result_roles = dict(map(
            lambda x: (x[0], AbcRole(
                id=x[0],
                display_name=x[1].get('name', {}).get('ru'),
                english_name=x[1].get('name', {}).get('en'),
            )),
            six.viewitems(result_roles),
        ))

        return result_roles, result_services_roles, result_scopes, result_services_scopes

    def get_all_persons_and_scopes(self, services=None):
        logger = logging.getLogger('info_logger')

        persons = {}
        try:
            members = self.get_service_members(
                filter_args={
                    'fields': ','.join((
                        'id',
                        'person',
                        'service',
                        'is_editable',
                        'role.id',
                        'role.name',
                        'role.scope.id',
                        'role.scope.name',
                        'role.scope.slug',
                    )),
                    'state': 'approved',
                    'service_state__in': 'develop,supported,needinfo',
                },
                oauth_token=self.config['abc']['oauth_token'],
            )
            # Достаем все роли и скоупы для всех сервисов
            roles, services_roles, scopes, services_scopes = self.get_roles_and_scopes(services=services)
        except BaseABCError as ex:
            raise ABCError(ex.message)

        for member in members:
            service_id = int(member['service']['id'])
            scope_id = int(member['role']['scope']['id'])
            role_id = int(member['role']['id'])
            if scope_id not in scopes:
                # На всякий случай вливаем неизвестные скоупы
                scopes[scope_id] = AbcScope(
                    id=scope_id,
                    unique_name=member['role']['scope']['slug'],
                    display_name=member['role']['scope'].get('name', {}).get('ru'),
                )
                logger.warning(
                    u'ABC: get_all_persons_and_scopes fetched an unknown members scope. Service id {}. Scope: {}'.format(
                        service_id,
                        str(member['role']['scope']),
                    ),
                )
            if role_id not in roles:
                # На всякий случай вливаем неизвестные роли
                roles[role_id] = AbcRole(
                    id=role_id,
                    display_name=member['role'].get('name', {}).get('ru'),
                    english_name=member['role'].get('name', {}).get('en'),
                )
                logger.warning(
                    u'ABC: get_all_persons_and_scopes fetched an unknown members role. Service id {}. Role: {}'.format(
                        service_id,
                        str(member['role']),
                    ),
                )

            if service_id not in services_scopes:
                services_scopes[service_id] = set()
            services_scopes[service_id].add(scope_id)

            if service_id not in services_roles:
                services_roles[service_id] = set()
            services_roles[service_id].add(role_id)

            uid = int(member['person']['uid'])
            persons.setdefault(uid, {})['login'] = member['person']['login']
            persons[uid].setdefault('group_ids', set()).add((service_id, scope_id))
            persons[uid].setdefault('roles_ids', set()).add((service_id, role_id))

        # Удаляем лишние связки сервисов, ролей и скоупов
        if services is not None:
            services_scopes = dict(
                filter(
                    lambda x: x[0] in services,
                    six.viewitems(services_scopes),
                ),
            )
            services_roles = dict(
                filter(
                    lambda x: x[0] in services,
                    six.viewitems(services_roles),
                ),
            )

        return persons, scopes, services_scopes, roles, services_roles

    def get_all_tvm_apps(self):
        """
        https://wiki.yandex-team.ru/intranet/abc/api/#get/api/v4/resources
        """
        params = {
            'fields': 'id,resource,service,state,state_display',
            'type': '47',  # 47 — TVM-приложение
            'page_size': self.tvm_apps_page_size,
        }
        results = {}
        try:
            tvm_apps = self._make_request_with_cursor_pagination(
                url_suffix='v4/resources/consumers/',
                method='GET',
                params=params,
                oauth_token=self.config['abc']['oauth_token'],
                page_size=self.tvm_apps_page_size,
            )
        except BaseABCError as ex:
            raise ABCError(ex.message)

        for app_data in tvm_apps:
            tvm_client_id = app_data['resource']['external_id']
            if tvm_client_id is None or tvm_client_id == 0:
                continue

            app_url = app_data['resource']['link']
            if app_url is not None and app_url.strip() == u'':
                app_url = None

            app = dict(
                tvm_client_id=tvm_client_id,
                name=app_data['resource']['name'],
                url=app_url,
                abc_resource_id=app_data['resource']['id'],
                abc_id=app_data['service']['id'],
                abc_state=app_data['state'],
                abc_state_display_name=app_data['state_display']['ru'],
            )
            results[tvm_client_id] = app

        return results


def get_abc(config):  # pragma: no cover
    return ABC(config)
