# -*- coding: utf-8 -*-

from passport.backend.core.builders import staff as passport_core_staff
from passport.backend.core.builders.staff.exceptions import BaseStaffError
from passport.backend.vault.api.builders.loggers import VaultGraphiteLogger
from passport.backend.vault.api.errors import StaffError


class Staff(passport_core_staff.Staff):
    page_size = 500

    def __init__(self, config, *args, **kwargs):
        super(Staff, self).__init__(
            reconnect_on_retries=True,
            graphite_logger=VaultGraphiteLogger(service='staff'),
            *args,
            **kwargs
        )
        self.config = config

    def _make_request_with_id_pagination(self, method='GET', url_suffix='', params=None, data=None,
                                         oauth_token=None, page_size=None,
                                         **kwargs):
        """
        Ходим по диапазонам айдишников с лимитом на размер страницы.
        """
        results = []
        page_size = page_size or self.page_size

        params = dict(params or {})
        params.update(
            _sort='id',
            _limit=page_size or self.page_size,
            _fields=','.join(
                filter(
                    lambda x: x != '',
                    [params.get('_fields', ''), 'id'],
                ),
            ),
        )

        min_id = 0
        while True:
            params.update(_query='id>{}'.format(min_id))
            rv = self._make_request(
                url_suffix=url_suffix,
                method=method,
                params=params,
                data=data,
                oauth_token=oauth_token,
                **kwargs
            )
            rv_results = rv['result']
            results += rv_results

            len_results = len(rv_results)
            if len_results == 0 or len_results < page_size:
                break

            min_id = rv_results[-1]['id']

        return results

    def get_all_departments(self, is_deleted=False):
        params = {
            '_fields': ','.join([
                'id',
                'url',
                'name',
                'is_deleted',
            ]),
            'is_deleted': 'true' if is_deleted else 'false',
            'type': 'department',
        }
        results = {}
        try:
            rv = self._make_request_with_id_pagination(
                url_suffix='/v3/groups',
                method='GET',
                params=params,
                oauth_token=self.config['staff']['oauth_token'],
            )
        except passport_core_staff.BaseStaffError as ex:
            raise StaffError(ex.message)
        for department in rv:
            department_id = department.get('id')
            if department_id is not None:
                department_id = int(department_id)
                results[department_id] = {
                    'url': department.get('url'),
                    'name': department.get('name'),
                    'is_deleted': department.get('is_deleted'),
                }
        return results

    def get_all_persons(self, is_dismissed=False):
        params = {
            '_fields': ','.join([
                'department_group.ancestors.id',
                'department_group.id',
                'uid',
                'login',
                'keys',
                'name.first.ru',
                'name.last.ru',
                'official.is_dismissed',
                'is_deleted',
            ]),
            'official.is_dismissed': 'true' if is_dismissed else 'false',
        }
        results = {}
        try:
            rv = self._make_request_with_id_pagination(
                url_suffix='/v3/persons',
                method='GET',
                params=params,
                oauth_token=self.config['staff']['oauth_token'],
            )
        except BaseStaffError as ex:
            raise StaffError(ex.message)
        for person in rv:
            uid = int(person['uid'])
            keys = list(map(lambda x: x['key'], person['keys']))
            results[uid] = {
                'group_ids': [],
                'keys': keys,
                'login': person['login'],
                'first_name': person['name']['first']['ru'],
                'last_name': person['name']['last']['ru'],
                'staff_id': person['department_group']['id'],
                'is_dismissed': person['official']['is_dismissed'],
                'is_deleted': person['is_deleted'],
            }
            results[uid]['_disabled'] = (results[uid]['is_dismissed'] or results[uid]['is_deleted'])
            results[uid]['group_ids'].append(person['department_group']['id'])
            results[uid]['group_ids'] += list(map(
                lambda ancestor: ancestor['id'],
                person['department_group']['ancestors'],
            ))
        return results


def get_staff(config):  # pragma: no cover
    return Staff(config)
