# coding: utf-8
import logging

import click
from flask.cli import (
    FlaskGroup,
    routes_command,
    shell_command,
    with_appcontext,
)
from flask_migrate.cli import db as db_commands
from passport.backend.library.wsgi_runner import Runner
from passport.backend.vault.api.db import upgrade_db
import yaml
import yenv

from .download import create_download_cli
from .find import create_find_cli
from .generate import create_generate_cli
from .golovan import create_golovan_cli
from .grants import create_grants_cli
from .roles import create_roles_cli
from .service import create_service_cli


info_logger = logging.getLogger('info_logger')
FORBIDDEN_FLASK_LOG_OPTIONS = ('SQLALCHEMY_BINDS', 'SQLALCHEMY_DATABASE_URI')


def log_config(config):
    info_logger.info(
        'Passport configured with the following settings:\n%s' % yaml.dump(
            config.get_config_by_basename('export.yaml'),
            default_flow_style=False,
        )
    )
    info_logger.info('Gunicorn has the following settings:\n%s' % '\n'.join(
        map(lambda x: '{}: {}'.format(*x), config.get('gunicorn', {}).items())
    ))
    info_logger.info('Flask configured with the following settings:\n%s' % yaml.dump(
        dict(filter(
            lambda x: x[0].upper() not in FORBIDDEN_FLASK_LOG_OPTIONS,
            config.get_config_by_basename('flask.yaml').items()
        )),
        default_flow_style=False,
    ))


def create_cli(app, config):
    cli = FlaskGroup(add_default_commands=False, create_app=lambda: app)
    cli.add_command(routes_command)
    cli.add_command(shell_command)
    if yenv.type == 'development':
        cli.add_command(db_commands)

    cli.add_command(create_download_cli(app, config))
    cli.add_command(create_find_cli(app, config))
    cli.add_command(create_roles_cli(app, config))
    cli.add_command(create_generate_cli(app, config))
    cli.add_command(create_grants_cli(app, config))
    cli.add_command(create_golovan_cli(app, config))

    @cli.command()
    @click.option('--skip-migrations', is_flag=True, default=False)
    @with_appcontext
    def run(skip_migrations):
        """Run server"""
        log_config(config)
        if not skip_migrations and config['application']['migrate']:
            upgrade_db(app)
        Runner(app, config).run()

    cli.add_command(create_service_cli(app, config))

    return cli
