# coding: utf-8

from contextlib import contextmanager
import logging
import sys

import click
from flask.cli import (
    AppGroup,
    with_appcontext,
)
from passport.backend.core.builders.xunistater import (
    get_xunistater,
    XunistaterBaseError,
)
from passport.backend.utils import p3
from passport.backend.utils.errors import unsafe
from passport.backend.utils.lock import lock
from passport.backend.vault.api.builders import (
    get_abc,
    get_staff,
)
from passport.backend.vault.api.models.external_record import ExternalRecord
from passport.backend.vault.api.models.tvm_app_info import TvmAppInfo
from passport.backend.vault.api.stats.golovan import make_xunistater_metrics
import six
import yenv


DOWNLOAD_ABC_LOCK_NAME = 'download_abc'
DOWNLOAD_ABC_LOCK_TIMEOUT = 500

DOWNLOAD_STAFF_LOCK_NAME = 'download_staff'
DOWNLOAD_STAFF_LOCK_TIMEOUT = 500

DOWNLOAD_TVM_APPS_LOCK_NAME = 'download_tvm_apps'
DOWNLOAD_TVM_APPS_LOCK_TIMEOUT = 500


class SkipDownload(Exception):
    pass


@contextmanager
def write_exception_to_log(logger, prefix=None):
    try:
        yield
    except Exception as e:
        prefix = six.u(prefix) + u' ' if prefix is not None else u''
        logger.info(u'{}{}'.format(prefix, p3.u8(str(e))))
        raise


def push_metrics_to_xunistater(
    config, logger, prefix, metrics,
    suffix='axxx', log_prefix='',
):
    if log_prefix:
        log_prefix = log_prefix.strip() + ': '

    logger.info('{}push metrics to the Xunistater'.format(log_prefix))
    with unsafe([XunistaterBaseError]) as push_result:
        xunistater_result = get_xunistater().push_metrics(
            make_xunistater_metrics(
                config,
                dict([
                    ('{}_{}_{}'.format(prefix, k, suffix), v)
                    for k, v in metrics.items()
                ])
            ),
        )

        logger.info(u'{}xunistater result: {}'.format(log_prefix, xunistater_result))

    if push_result['status'] is False:
        logger.info(u'{}xunistater failed: {}'.format(log_prefix, push_result))


def create_download_cli(app, config):
    download_cli = AppGroup('download', help='Download external data from ABC and Staff to cache')

    def check_cron_environment(logging_prefix, logger):
        if yenv.type not in config['download']['cron']['environments']:
            logger.info(
                '{}: The command cannot be invoked in the "{}" environment'.format(
                    logging_prefix,
                    yenv.type,
                )
            )
            sys.exit(3)

    @download_cli.command()
    @with_appcontext
    @click.option('--yasm-ttl', 'ttl', help='Metrics ttl value', type=click.INT, default=None)
    @click.option('--cron', 'cron', is_flag=True, help='Check environment')
    def abc(ttl, cron):
        """Fetch data from the ABC"""
        logger = logging.getLogger('info_logger')
        if cron:
            check_cron_environment('ABC', logger)

        metrics = dict(
            fetched=0,
            errors=0,
            new_external_records=0,
            old_external_records=0,
            new_department_infos=0,
            old_department_infos=0,
        )

        with write_exception_to_log(logger, 'ABC error:'):
            try:
                with lock(
                    config['ylock'], DOWNLOAD_ABC_LOCK_NAME,
                    skip_lock=not cron, timeout=DOWNLOAD_ABC_LOCK_TIMEOUT,
                    logger=logger, log_prefix='ABC: '
                ) as is_locked_successfully:
                    if is_locked_successfully is False:
                        raise SkipDownload('ABC: skip download')

                    abc = get_abc(config)
                    logger.info('ABC: started downloading services...')
                    abc_departments = abc.get_all_departments()

                    logger.info('ABC: started downloading persons and scopes...')
                    (
                        abc_persons,
                        abc_scopes,
                        abc_services_scopes,
                        abc_roles,
                        abc_services_roles,
                    ) = abc.get_all_persons_and_scopes(
                        services=abc_departments,
                    )

                    result = ExternalRecord.insert_abc(
                        abc_persons,
                        abc_scopes,
                        abc_services_scopes,
                        abc_departments,
                        abc_roles,
                        abc_services_roles,
                    )

                    metrics['fetched'] = 1
                    metrics['new_external_records'] = result['new_external_records']
                    metrics['old_external_records'] = result['old_external_records']
                    metrics['new_department_infos'] = result['new_department_infos']
                    metrics['old_department_infos'] = result['old_department_infos']

            except SkipDownload as e:
                logger.info(str(e))
            except:
                metrics['errors'] = 1
                raise
            finally:
                push_metrics_to_xunistater(config, logger, 'abc', metrics, log_prefix='ABC')

    @download_cli.command()
    @with_appcontext
    @click.option('--yasm-ttl', 'ttl', help='Metrics ttl value', type=click.INT, default=None)
    @click.option('--cron', 'cron', is_flag=True, help='Check environment')
    def staff(ttl, cron):
        """Fetch data from the Staff"""
        logger = logging.getLogger('info_logger')
        if cron:
            check_cron_environment('Staff', logger)

        metrics = dict(
            fetched=0,
            errors=0,
            new_external_records=0,
            old_external_records=0,
            new_user_infos=0,
            new_department_infos=0,
        )

        with write_exception_to_log(logger, 'Staff error:'):
            try:
                with lock(
                    config['ylock'], DOWNLOAD_STAFF_LOCK_NAME,
                    skip_lock=not cron, timeout=DOWNLOAD_STAFF_LOCK_TIMEOUT,
                    logger=logger, log_prefix='Staff: '
                ) as is_locked_successfully:
                    if is_locked_successfully is False:
                        raise SkipDownload('Staff: skip download')

                    staff = get_staff(config)

                    logger.info('Staff: started downloading departments...')
                    staff_departments = staff.get_all_departments(is_deleted=True)
                    staff_departments.update(staff.get_all_departments(is_deleted=False))

                    logger.info('Staff: started downloading persons...')
                    staff_persons = staff.get_all_persons(is_dismissed=True)
                    staff_persons.update(staff.get_all_persons(is_dismissed=False))

                    result = ExternalRecord.insert_staff(
                        staff_persons,
                        staff_departments,
                    )

                    metrics['fetched'] = 1
                    metrics['new_external_records'] = result['new_external_records']
                    metrics['old_external_records'] = result['old_external_records']
                    metrics['new_user_infos'] = result['new_user_infos']
                    metrics['new_department_infos'] = result['new_department_infos']
            except SkipDownload as e:
                logger.info(str(e))
            except:
                metrics['errors'] = 1
                raise
            finally:
                push_metrics_to_xunistater(config, logger, 'staff', metrics, log_prefix='Staff')

    @download_cli.command('tvm_apps')
    @with_appcontext
    @click.option('--yasm-ttl', 'ttl', help='Metrics ttl value', type=click.INT, default=None)
    @click.option('--cron', 'cron', is_flag=True, help='Check environment')
    def tvm_apps(ttl, cron):
        """Fetch TVM apps data from the ABC"""
        logger = logging.getLogger('info_logger')
        if cron:
            check_cron_environment('TVM apps', logger)

        metrics = dict(
            fetched=0,
            errors=0,
            download_apps=0,
        )

        with write_exception_to_log(logger, 'TVM apps error:'):
            try:
                with lock(
                    config['ylock'], DOWNLOAD_TVM_APPS_LOCK_NAME,
                    skip_lock=not cron, timeout=DOWNLOAD_STAFF_LOCK_TIMEOUT,
                    logger=logger, log_prefix='TVM apps: '
                ) as is_locked_successfully:
                    if is_locked_successfully is False:
                        raise SkipDownload('TVM apps: skip download')

                    abc = get_abc(config)

                    logger.info('TVM apps: started downloading apps...')
                    tvm_apps = abc.get_all_tvm_apps()
                    result = TvmAppInfo.insert_tvm_apps(tvm_apps)

                    metrics['fetched'] = 1
                    metrics['download_apps'] = result['tvm_apps']
            except SkipDownload as e:
                logger.info(str(e))
            except:
                metrics['errors'] = 1
                raise
            finally:
                push_metrics_to_xunistater(config, logger, 'tvm_apps', metrics, log_prefix='TVM apps')

    return download_cli
