# coding: utf-8

import sys
import textwrap

import click
from flask.cli import (
    AppGroup,
    with_appcontext,
)
from passport.backend.vault.api.models import AbcDepartmentInfo
from passport.backend.vault.utils.roles import format_role
from tabulate import tabulate
import yenv


def create_find_cli(app, config):
    find_cli = AppGroup('find', help='Find objects')

    @find_cli.command()
    @click.argument('abc_service_id_or_slug', required=True)
    @with_appcontext
    def abc(abc_service_id_or_slug):
        """Find ABC service"""
        click.echo(u'Environment: {}'.format(yenv.type))
        click.echo()

        service = find_abc_service(abc_service_id_or_slug)
        if not service:
            click.echo(u'Service "{}" not found'.format(abc_service_id_or_slug))
            sys.exit(1)

        print_abc_service(service)

    return find_cli


def find_abc_service(abc_service_id_or_slug):
    abc_service_id_or_slug = abc_service_id_or_slug.strip()
    result = None

    if abc_service_id_or_slug.isdigit():
        result = AbcDepartmentInfo.query.filter(
            AbcDepartmentInfo.id == abc_service_id_or_slug,
        ).one_or_none()

    if result is None:
        result = AbcDepartmentInfo.query.filter(
            AbcDepartmentInfo.unique_name == abc_service_id_or_slug,
        ).one_or_none()

    return result


def print_abc_service(service):
    click.echo(textwrap.dedent(
        u'''
        Service: {service.display_name}
        ABC ID: {service.id}
        ABC slug: {service.unique_name}
        '''.format(
            service=service,
        )
    ).strip())
    click.echo()

    scopes = list(service.scopes)
    if scopes:
        click.echo('Scopes:')
        click.echo(
            tabulate(
                map(
                    lambda x: [
                        format_role(dict(
                            role_slug='owner',
                            abc_id=service.id,
                            abc_scope=x.unique_name,
                        )),
                        x.display_name,
                    ],
                    scopes,
                ),
                tablefmt='psql',
            ),
        )
        click.echo()

    roles = list(service.roles)
    if roles:
        click.echo('Roles:')
        click.echo(
            tabulate(
                map(
                    lambda x: [
                        format_role(dict(
                            role_slug='owner',
                            abc_id=service.id,
                            abc_role=x.id,
                        )),
                        x.display_name,
                    ],
                    roles,
                ),
                tablefmt='psql',
            ),
        )
        click.echo()
