# coding: utf-8

import sys

import click
from flask.cli import (
    AppGroup,
    with_appcontext,
)
from passport.backend.vault.api import models
from passport.backend.vault.api.db import get_db
from passport.backend.vault.api.utils.secrets import token_urlsafe
import sqlalchemy.dialects as sql_dialects
from sqlalchemy.schema import (
    CreateIndex,
    CreateTable,
)


def generate_key():
    return token_urlsafe(18)


def ym_iter(start_year, start_month, end_year, end_month):
    ym_start = 12 * start_year + start_month - 1
    ym_end = 12 * end_year + end_month - 1
    for ym in range(ym_start, ym_end):
        y, m = divmod(ym, 12)
        yield y, m + 1


def sql_dialect_by_name(name):
    sql_dialects.registry.load(name)
    return getattr(sql_dialects, name).dialect()


def sql_schema_for_table(table, dialect):
    result = []
    result.append(str(CreateTable(table).compile(dialect=dialect)).strip() + ';')
    for index in table.indexes:
        result.append(str(CreateIndex(index).compile(dialect=dialect)).strip() + ';')
    return result


def create_generate_cli(app, config):
    generate_cli = AppGroup('generate', help='Generate SQL statements for migrations')

    @generate_cli.command()
    @with_appcontext
    @click.option('--dialect', help='sql dialect (mysql, sqlite)', default='mysql')
    def schema(dialect):
        """Generate a full sql-schema"""
        try:
            dialect = sql_dialect_by_name(dialect)
        except Exception:
            click.echo('Dialect "{}" not found'.format(dialect), err=True)
            sys.exit(1)

        metadata = get_db().metadata
        for table in metadata.sorted_tables:
            for line in sql_schema_for_table(table, dialect):
                click.echo(line)

    @generate_cli.command('sql_for')
    @with_appcontext
    @click.argument('model')
    @click.option('--dialect', help='sql dialect (mysql, sqlite)', default='mysql')
    def sql_for(model, dialect):
        """Generate sql for a model"""
        try:
            dialect = sql_dialect_by_name(dialect)
        except Exception:
            click.echo('Dialect "{}" not found'.format(dialect), err=True)
            sys.exit(1)

        if model and hasattr(models, model):
            for line in sql_schema_for_table(getattr(models, model).__table__, dialect):
                click.echo(line)
        else:
            click.echo('Model "{}" not found.'.format(model), err=True)
            sys.exit(1)

    @generate_cli.command()
    @with_appcontext
    @click.option('--start-date', help='start year and month in yyyymm format', default='201801')
    @click.option('--end-date', help='end year and month in yyyymm format', default='203001')
    def keys(start_date, end_date):
        """Generate cipher keys"""
        start_year, start_month = int(start_date[:4]), int(start_date[4:])
        end_year, end_month = int(end_date[:4]), int(end_date[4:])

        for year, month in ym_iter(start_year, start_month, end_year, end_month):
            click.echo('%d%02d %s\n' % (year, month, generate_key()), nl=False)

    return generate_cli
