# coding: utf-8

import json

import click
from flask import g
from flask.cli import (
    AppGroup,
    with_appcontext,
)
from passport.backend.core.builders.xunistater import (
    get_xunistater,
    XunistaterBaseError,
)
from passport.backend.core.builders.yasm_agent import (
    get_yasm_agent,
    YasmAgentBaseError,
)
from passport.backend.utils.errors import unsafe
from passport.backend.vault.api.stats.golovan import (
    get_global_metrics,
    make_golovan_metrics,
    make_xunistater_metrics,
)


def create_golovan_cli(app, config):
    golovan_cli = AppGroup('golovan', help='Generate monitoring data for Golovan')

    @golovan_cli.command('global')
    @with_appcontext
    @click.option('--ttl', help='Metrics ttl value', type=click.INT, default=None)
    @click.option('--print', 'print_data', is_flag=True, help='Print json to stdout instead push to the Yasm Agent')
    def global_(ttl, print_data):
        """Push global metrics to the local Yasm Agent"""
        g.use_slave = True

        metrics = get_global_metrics()
        if print_data:
            click.echo(
                json.dumps(metrics, sort_keys=True, indent=True),
            )
        else:
            with unsafe([YasmAgentBaseError]):
                click.echo('Push to the YASM Agent: {}'.format(
                    get_yasm_agent().push_metrics(
                        make_golovan_metrics(
                            config,
                            dict([(k + '_max', v) for k, v in metrics.items()]),
                            ttl=ttl
                        ),
                    )
                ))

            with unsafe([XunistaterBaseError]):
                click.echo('Push to the Xunistater: {}'.format(
                    get_xunistater().push_metrics(
                        make_xunistater_metrics(
                            config,
                            dict([(k + '_axxx', v) for k, v in metrics.items()]),
                        ),
                    )
                ))

    return golovan_cli
