# coding: utf-8

import logging

import click
from flask.cli import (
    AppGroup,
    with_appcontext,
)
from passport.backend.vault.api.errors import TvmGrantExistsError
from passport.backend.vault.api.models import TvmGrants
from tabulate import tabulate
import yenv

from .base import (
    format_timestamp,
    format_tvm_app,
    get_fqdn,
    get_user_login,
)


def create_grants_cli(app, config):
    grants_cli = AppGroup('grants', help='Manage Vault API grants')

    def statbox_log(logger, action, **kwargs):
        log_data = dict(kwargs)
        log_data.update(dict(
            mode='tvm_grants',
            action=action,
            login=get_user_login(),
            fqdn=get_fqdn(),
        ))
        logger.info(log_data)

    @grants_cli.command()
    @with_appcontext
    def list():
        """Show grants list"""
        logger = logging.getLogger('statbox')

        table = map(
            lambda x: [
                x.tvm_client_id,
                format_tvm_app(x.tvm_app),
                x.comment,
                format_timestamp(x.created_at),
            ],
            TvmGrants.list(),
        )

        click.echo('Vault API grants list')
        click.echo('Environment: {}'.format(yenv.type))
        click.echo()
        click.echo(
            tabulate(
                table,
                headers=['TVM client ID', 'Application', 'Comment', 'Created at'],
                tablefmt='psql'
            )
        )
        statbox_log(logger, u'list')

    @grants_cli.command()
    @click.argument('tvm_client_id', type=click.IntRange(min=1), required=True)
    @click.argument('comment', required=False)
    @with_appcontext
    def grant(tvm_client_id, comment=None):
        """Grant access for TVM app"""
        logger = logging.getLogger('statbox')

        click.echo(u'Environment: {}'.format(yenv.type))
        click.echo()

        click.echo(u'TVM client ID: {}'.format(tvm_client_id))
        click.echo(u'Comment: {}'.format(comment or ''))
        try:
            TvmGrants.grant(tvm_client_id, comment)
            click.echo(u'Access granted')
            statbox_log(logger, u'grant', tvm_client_id=tvm_client_id)
        except TvmGrantExistsError:
            click.echo(u'Grant is already exists')

    @grants_cli.command()
    @click.argument('tvm_client_id', type=click.IntRange(min=1), required=True)
    @with_appcontext
    def revoke(tvm_client_id):
        """Revoke access for TVM app"""
        logger = logging.getLogger('statbox')

        click.echo(u'Environment: {}'.format(yenv.type))
        click.echo()

        click.echo(u'TVM client ID: {}'.format(tvm_client_id))

        grant = TvmGrants.fetch(tvm_client_id)
        if grant is None:
            click.echo(u'Grant is not exists')
        else:
            TvmGrants.revoke(tvm_client_id)
            statbox_log(logger, u'revoke', tvm_client_id=tvm_client_id)
            click.echo(u'Access revoked')

    return grants_cli
