# coding: utf-8

import logging
import sys
import textwrap

import click
from flask.cli import (
    AppGroup,
    with_appcontext,
)
from passport.backend.utils.errors import unsafe
from passport.backend.vault.api.errors import NonexistentEntityError
from passport.backend.vault.api.models import (
    AbcDepartmentInfo,
    Roles,
    Secret,
    StaffDepartmentInfo,
    UserInfo,
    UserRole,
)
from passport.backend.vault.api.utils.errors import BaseUUIDError
from passport.backend.vault.utils.roles import (
    format_role,
    RoleAction,
)
from tabulate import tabulate
import yenv

from .base import (
    get_fqdn,
    get_user_login,
)


def create_roles_cli(app, config):
    roles_cli = AppGroup('roles', help='Manage secrets roles')

    def statbox_log(logger, action, **kwargs):
        log_data = dict(kwargs)
        if 'role' in log_data:
            log_data['role'] = str(log_data['role'])

        log_data.update(dict(
            mode='cli_roles',
            action=action,
            login=get_user_login(),
            fqdn=get_fqdn(),
        ))
        logger.info(log_data)

    def get_creator_uid():
        creator = None
        user_login = get_user_login()
        if user_login and user_login != 'root':
            with unsafe([NonexistentEntityError]):
                creator = UserInfo.get_by_login(user_login)

        robot_login = config.get('cli', {}).get('robot_login')
        if not creator and robot_login:
            with unsafe([NonexistentEntityError]):
                creator = UserInfo.get_by_login(robot_login)

        return creator.uid or 0

    @roles_cli.command()
    @click.argument('secret_uuid', required=True)
    @with_appcontext
    def info(secret_uuid):
        """Get secrets roles"""
        logger = logging.getLogger('statbox')

        click.echo(u'Environment: {}'.format(yenv.type))
        click.echo()

        secret = find_secret(secret_uuid)
        statbox_log(logger, u'info', secret_uuid=secret_uuid)

        print_secret(secret)

    @roles_cli.command()
    @click.argument('secret_uuid', required=True)
    @click.argument('role', required=True)
    @with_appcontext
    def add(secret_uuid, role):
        """
        Add a user role to the secret

        Role format: (reader|owner|appender):(user|abc|staff):(id|slug)[:(scope:abc_scope|role:abc_role_id)]

        Examples: owner:user:arhibot, reader:staff:yandex, reader:abc:123:scope:administration, reader:abc:123:role:630
        """
        logger = logging.getLogger('statbox')

        click.echo(u'Environment: {}'.format(yenv.type))
        click.echo()

        secret = find_secret(secret_uuid)
        parsed_role = parse_role(role)
        creator_uid = get_creator_uid()

        added, _ = add_secret_role(secret, parsed_role, creator_uid)
        if added:
            statbox_log(
                logger,
                u'add_role',
                secret_uuid=secret_uuid,
                creator_uid=creator_uid,
                raw_role=role,
                **parsed_role
            )

        secret = Secret.query.get(secret.uuid)
        print_secret(secret)

    return roles_cli


def find_secret(secret_uuid):
    try:
        secret = Secret.query.get(secret_uuid)
    except BaseUUIDError as e:
        click.echo(e.msg)
        sys.exit(1)

    if secret is None:
        click.echo('Secret "{}" not found'.format(secret_uuid))
        sys.exit(1)

    return secret


def parse_role(user_role):
    try:
        role = RoleAction.from_string(user_role)
        role = dict(
            role=Roles[role.role_type.upper()],
            **role.as_method_params()
        )

        if 'staff_id' in role and not role['staff_id'].isdigit():
            role['staff_id'] = StaffDepartmentInfo.get_by_unique_name(role['staff_id']).id

        if 'abc_id' in role and not role['abc_id'].isdigit():
            role['abc_id'] = AbcDepartmentInfo.get_by_unique_name(role['abc_id']).id

        if 'login' in role:
            role['uid'] = UserInfo.get_by_login(role['login'], with_keys=False).uid
            del role['login']

        return role
    except Exception as e:
        click.echo(str(e))
        sys.exit(1)


def add_secret_role(secret, role, creator_uid):
    try:
        user_role = UserRole.create_user_role(
            creator_uid=creator_uid,
            secret=secret,
            **role
        )
        return UserRole.safely_add_user_role_to_secret(user_role)
    except Exception as e:
        click.echo(str(e))
        sys.exit(1)


def print_secret(secret):
    click.echo(textwrap.dedent(
        u'''
        Secret UUID: {secret.uuid}
        Name: {secret.name}
        '''.format(secret=secret)
    ).strip())
    click.echo()

    click.echo('Roles:')
    click.echo(
        tabulate(
            [
                [r.role_slug().lower(), format_role(r.serialize())]
                for r in secret.secret_roles
            ],
            tablefmt='psql',
        )
    )
