# -*- coding: utf-8 -*-

import re

from passport.backend.utils.common import inheritors
from passport.backend.vault.api.utils.errors import BaseError
from sqlalchemy.exc import IntegrityError


# Базовые ошибки


class AccessError(BaseError):
    """
    Отсутствуют доступы для совершения запрошенной операции
    """
    status_code = 401
    code = 'access_error'
    msg = 'Access denied'


class ExternalServiceError(BaseError):
    """
    Произошла ошибка во внешнем сервисе
    """
    status_code = 500
    code = 'external_service_error'
    msg = 'External service error'
    level = 'ERROR'


class ValidationError(BaseError):
    """
    Ошибка валидации переданных данных
    """
    status_code = 400
    code = 'validation_error'
    msg = 'Validation error'

    def __init__(self, errors):
        self.errors = errors
        self.kwargs = {'errors': errors}


# Конкретные ошибки

class PingfileError(BaseError):
    """
    Нельзя прочитать pingfile в ручке /ping.html
    """
    status_code = 503
    code = 'out_of_service'
    msg = 'Pingfile is not readable'


class PingDatabaseError(BaseError):
    """
    Ошибка при обращении к БД в ручке /ping.html
    """
    status_code = 500
    code = 'ping_failed'
    msg = 'Database failed'
    level = 'ERROR'


class ABCError(ExternalServiceError):
    """
    Обертка для ошибки ABC
    """
    status_code = 500
    code = 'abc_error'
    msg = 'ABC server error'


class ClientError(BaseError):
    """
    Базовая ошибка клиента Vault
    """
    def __str__(self):
        return str(self.kwargs)


class DecryptionError(BaseError):
    """
    Ошибка расшифровки данных из БД
    """
    status_code = 500
    code = 'decryption_error'
    msg = 'Error when decrypting the data from the database'
    level = 'ERROR'


class DelegationAccessError(AccessError):
    """
    Ошибка доступа по делегирующему токену
    """
    status_code = 401
    code = 'delegation_access_error'
    msg = 'Access error by delegation token'


class DelegationTokenMismatchTvmAppError(DelegationAccessError):
    """
    Ошибка доступа по делегирующему токену (не совпадает TVM-приложение)
    """
    msg = 'Access error by delegation token'
    info = {'reason': 'The token was issued to another TVM application'}


class InvalidDelegationTokenSignatureError(DelegationAccessError):
    """
    Ошибка доступа по делегирующему токену (не совпадает подпись)
    """
    msg = 'Access error by delegation token'
    info = {'reason': 'Does not match the signature of the token'}


class DelegationTokenDoesNotMatchSecretError(AccessError):
    """
    Токен выписан для другого секрета
    """
    status_code = 400
    code = 'token_does_not_match_secret'
    msg = 'Token does not match secret uuid'


class DelegationTokenRevokedError(AccessError):
    """
    Делегирующий токен отозван
    """
    status_code = 410
    code = 'token_revoked'
    msg = 'Delegation token revoked'


class InvalidDelegationTokenError(BaseError):
    """
    У токена невалидная подпись
    """
    status_code = 400
    code = 'invalid_token'
    msg = 'Invalid delegation token'


class TokenNotAssociatedWithTvmAppError(AccessError):
    """
    Делегирующий токен не связан с TVM-приложением
    """
    status_code = 412
    code = 'token_not_associated_with_tvm_app'
    msg = 'Delegation token is not associated with the tvm-app'


class ExactlyOneFieldError(ValidationError):
    """
    Ошибка валидации. Требуется передача только одного поля из обозначенного списка
    """
    def __init__(self, fields):
        all_fields = ', '.join(fields)
        errors = {field: 'Only one parameter must be set among %s' % all_fields for field in fields}
        super(ExactlyOneFieldError, self).__init__(errors)


def mask_oauth_token(token):
    if not token:
        return

    m = re.search(r'(?i)(?:oauth\s*)?(\S+)', token)
    if m:
        token = m.group(1)
        l = len(token) // 2
        return token[:l] + '*' * l
    return


class BaseOAuthError(AccessError):
    def __init__(self, oauth_token=None, private_info=None, **kwargs):
        private_info = dict(private_info) if private_info is not None else {}
        if oauth_token:
            private_info['oauth_token'] = mask_oauth_token(oauth_token)

        super(BaseOAuthError, self).__init__(
            private_info=private_info,
            **kwargs
        )


class InvalidOauthTokenError(BaseOAuthError):
    """
    Передан неверный OAuth токен
    """
    code = 'invalid_oauth_token_error'
    msg = 'Invalid oauth token'


class InvalidScopesError(BaseOAuthError):
    code = 'invalid_scopes_error'
    msg = 'Invalid scope in a token'


class LastOwnerError(BaseError):
    """
    Невозможно совершить операцию, так как у сущности после ее выполнения не останется владельца
    """
    status_code = 400
    code = 'last_owner_error'
    msg = 'It is impossible to perform operation as entity would become orphan'


class LoginHeaderInRsaSignatureRequiredError(BaseError):
    """
    Требуется заголовок с логином при передаче RSA подписи
    """
    status_code = 400
    code = 'login_header_in_rsa_signature_required_error'
    msg = 'Required header with the username when passing the RSA signature'


class MaximumFieldLengthExceededError(ValidationError):
    def __init__(self, field_name, current_length, maximum_length):
        self.current_length = current_length
        self.maximum_length = maximum_length

        errors = {
            field_name: [
                'length: 1_to_%d' % self.maximum_length,
                'Value length exceeds %d' % self.maximum_length,
            ],
        }
        super(MaximumFieldLengthExceededError, self).__init__(errors=errors)


class EmptyDiffSecretVersionError(BaseError):
    """
    После вычисления diff'а секрет не содержит ключей
    """
    status_code = 400
    code = 'empty_diff_secret_version'
    msg = 'Secret version has no keys. You cannot remove all keys from a secret version'


class NonexistentEntityError(BaseError):
    """
    Запрошена несуществующая сущность
    """
    status_code = 400
    code = 'nonexistent_entity_error'
    msg = 'Requested a non-existent entity'

    def __init__(self, entity_class=None, entity_id=None):
        self.kwargs = {}
        if entity_class is not None:
            if isinstance(entity_class, (list, tuple)):
                entity_class = ', '.join(map(lambda x: x.__name__, entity_class))
            else:
                entity_class = entity_class.__name__
            self.kwargs['class'] = entity_class

        if isinstance(entity_id, (list, tuple)):
            entity_id = ', '.join(entity_id)

        if entity_id:
            self.kwargs['id'] = entity_id

        if entity_class is not None and entity_id is not None:
            self.msg = self.msg + ' ({}, {})'.format(entity_class, entity_id)


class OmgUuidCollisionError(BaseError):
    """
    У нас UUID совпали, сворачиваемся
    """
    status_code = 500
    code = 'omg_uuid_collision_error'
    level = 'ERROR'


class OutdatedRsaSignatureError(BaseError):
    """
    RSA подпись исткела
    """
    status_code = 400
    code = 'outdated_rsa_signature_error'
    msg = 'Outdated RSA signature. Check and synchronize the local time on the computer'


class ServiceTicketParsingError(AccessError):
    """
    Ошибка проверки сервисного тикета
    """
    status_code = 401
    code = 'service_ticket_parsing_error'
    msg = 'Invalid service ticket'


class ServiceTicketRequiredError(AccessError):
    """
    Требуется сервисный тикет
    """
    status_code = 401
    code = 'service_ticket_required_error'
    msg = 'Service ticket required'


class TvmGrantRequiredError(AccessError):
    """
    Требуется грант для tvm-приложения
    """
    code = 'tvm_grant_required_error'

    def __init__(self, tvm_client_id, *args, **kwargs):
        super(TvmGrantRequiredError, self).__init__(
            msg='Requires a grant to the tvm application (client_id: {})'.format(tvm_client_id),
            *args,
            **kwargs
        )


class TvmGrantExistsError(BaseError, IntegrityError):
    """
    Грант на TVM-приложение уже выдан
    """
    code = 'tvm_grant_exists_error'
    msg = 'TVM grant exists'


class StaffError(ExternalServiceError):
    """
    Обертка для ошибки Staff
    """
    status_code = 500
    code = 'staff_error'
    msg = 'Staff server error'


class RsaSignatureError(AccessError):
    """
    Ошибка проверки RSA
    """
    status_code = 401
    code = 'rsa_signature_error'
    msg = 'RSA signature error'


class TimestampHeaderInRsaSignatureRequiredError(BaseError):
    """
    Требуется заголовок со временем для RSA подписи
    """
    status_code = 400
    code = 'timestamp_header_in_rsa_signature_required_error'
    msg = 'Requires header with time for RSA signature'


class UserAuthRequiredError(AccessError):
    """
    Требуется авторизация пользователя
    """
    status_code = 401
    code = 'user_auth_required_error'
    msg = 'User auth required'


class UserTicketRequiredError(AccessError):
    """
    Требуется пользовательский тикет
    """
    status_code = 401
    code = 'user_ticket_required_error'
    msg = 'User ticket required'


class UnallowedTvmClientError(AccessError):
    """
    В данных условиях тикет с заданным tvm_client_id не подходит
    """
    status_code = 401
    code = 'unallowed_tvm_client_error'
    msg = 'Under these conditions, a ticket with a specified tvm_client_id not suitable'


class UserTicketParsingError(AccessError):
    """
    Ошибка проверки пользовательского тикета
    """
    status_code = 401
    code = 'user_ticket_parsing_error'
    msg = 'Invalid user ticket'


class ZeroDefaultUidError(AccessError):
    """
    Пришли с юзер тикетом, у которого default_uid = 0
    """
    status_code = 401
    code = 'zero_default_uid_error'
    msg = 'User ticket has default_uid equals to zero'


class SecretHasNewHeadVersionError(BaseError):
    """
    У секрета новая head-версия
    """
    status_code = 409
    code = 'secret_has_new_head_version'
    msg = 'Secret has a new head version'


class HeadVersionNotFoundError(BaseError):
    """
    У секрета отсутсвует head-версия
    """
    status_code = 404
    code = 'head_version_not_found'
    msg = 'Head version not found'


class AbcServiceNotFoundError(BaseError):
    """
    Не найден ABC-сервис
    """
    status_code = 400
    code = 'abc_service_not_found'

    def __init__(self, abc_id, *args, **kwargs):
        super(AbcServiceNotFoundError, self).__init__(
            msg='ABC service not found (abc_id: {})'.format(abc_id),
            *args,
            **kwargs
        )


class AbcScopeNotFoundError(BaseError):
    """
    Не найден скоуп для ABC-сервиса
    """
    status_code = 400
    code = 'abc_scope_not_found'

    def __init__(self, abc_id, abc_scope, *args, **kwargs):
        if abc_id is None:
            msg = 'Scope not found (abc_scope: "{}")'.format(abc_scope)
        else:
            msg = 'Scope not found in a service (abc_id: {}, abc_scope: "{}")'.format(abc_id, abc_scope)
        super(AbcScopeNotFoundError, self).__init__(msg=msg, *args, **kwargs)


class AbcRoleNotFoundError(BaseError):
    """
    Не найдена роль для ABC-сервиса
    """
    status_code = 400
    code = 'abc_role_not_found'

    def __init__(self, abc_id, abc_role_id, *args, **kwargs):
        if abc_id is None:
            msg = 'Role not found (abc_role_id: "{}")'.format(abc_role_id)
        else:
            msg = 'Role not found in a service (abc_id: {}, abc_role_id: "{}")'.format(abc_id, abc_role_id)
        super(AbcRoleNotFoundError, self).__init__(msg=msg, *args, **kwargs)


class StaffGroupNotFoundError(BaseError):
    """
    Не найдена Staff-группа
    """
    status_code = 400
    code = 'staff_group_not_found'

    def __init__(self, staff_id, *args, **kwargs):
        super(StaffGroupNotFoundError, self).__init__(
            msg='Staff group not found (staff_id: {})'.format(staff_id),
            *args,
            **kwargs
        )


class UserNotFoundError(BaseError):
    """
    Не найден пользователь
    """
    status_code = 400
    code = 'user_not_found'

    def __init__(self, uid=None, login=None, *args, **kwargs):
        msg = u'User not found'
        if uid is not None:
            msg += u' (uid: {})'.format(uid)
        if login is not None:
            msg += u' (login: {})'.format(login)

        super(UserNotFoundError, self).__init__(
            msg=msg,
            *args,
            **kwargs
        )


class DatabaseOperationalError(BaseError):
    """
    Обёртка для OperationalError базы данных
    """
    status_code = 503
    code = 'service_temporary_down'
    msg = 'Service temporary down'

    def __init__(self, exc, use_slave, *args, **kwargs):
        super(DatabaseOperationalError, self).__init__(
            private_info=dict(
                use_slave=use_slave,
                exception=str(exc),
            ),
        )


error_by_code = {x.code: x for x in inheritors(BaseError)}
