# coding: utf-8
"""
Add ABC scopes

Revision ID: ddb37aa51019
Revises: f00c7b3d54bc
Create Date: 2018-11-26 18:29:30.832956
"""

import textwrap
import time

from alembic import (
    context,
    op,
)
from passport.backend.vault.api import models
import sqlalchemy as sa


# revision identifiers, used by Alembic.
revision = 'ddb37aa51019'
down_revision = 'f00c7b3d54bc'
branch_labels = None
depends_on = None

ABC_DEVELOPMENT_SCOPE_ID = 5


def upgrade():
    is_mysql = (op.get_bind().dialect.name == 'mysql')

    op.create_table('abc_scopes',
        sa.Column('id', models.base.MagicInteger(), nullable=False),
        sa.Column('unique_name', sa.String(length=255), nullable=False),
        sa.Column('display_name', sa.String(length=255), nullable=False),
        sa.Column('created_at', models.base.Timestamp(), nullable=False),
        sa.PrimaryKeyConstraint('id'),
        sa.UniqueConstraint('unique_name', name='ix_abc_scopes_unique_name'),
    )

    op.execute(textwrap.dedent(
        u'''
            INSERT INTO abc_scopes (id, unique_name, display_name, created_at)
            VALUES (5, 'development', 'Разработка', {})
        '''.format(time.time())
    ))

    op.create_table('abc_departments_to_scopes',
        sa.Column('abc_department_info', models.base.MagicBigInteger(), nullable=False),
        sa.Column('abc_scopes', models.base.MagicInteger(), nullable=False),
        sa.ForeignKeyConstraint(['abc_department_info'], ['abc_department_info.id'], name='abc_departments_to_scopes_ibfk_1'),
        sa.ForeignKeyConstraint(['abc_scopes'], ['abc_scopes.id'], name='abc_departments_to_scopes_ibfk_2'),
        sa.PrimaryKeyConstraint('abc_department_info', 'abc_scopes'),
    )

    op.execute(textwrap.dedent(
        '''
            INSERT INTO abc_departments_to_scopes (abc_department_info, abc_scopes)
            SELECT id, 5 FROM abc_department_info
        '''
    ))

    with op.batch_alter_table('external_records') as batch_op:
        batch_op.add_column(
            sa.Column('external_scope_id', models.base.MagicInteger(), nullable=False, server_default='0'),
        )
        if is_mysql:
            batch_op.drop_constraint(None, type_='primary')
        batch_op.create_primary_key(
            None if is_mysql else 'primary_key',
            ('uid', 'external_type', 'external_id', 'external_scope_id'),
        )

    op.create_index(
        'idx_external_records_services',
        'external_records',
        ['external_type', 'external_id', 'external_scope_id'],
    )

    op.add_column(
        'user_roles',
        sa.Column('abc_scope_id', models.base.MagicInteger(), nullable=True),
    )

    op.add_column(
        'abc_department_info',
        sa.Column('state', sa.Integer(), server_default='0', nullable=False),
    )
    op.add_column(
        'staff_department_info',
        sa.Column('state', sa.Integer(), server_default='0', nullable=False),
    )
    op.add_column(
        'user_info',
        sa.Column('state', sa.Integer(), server_default='0', nullable=False),
    )

    with op.batch_alter_table('user_roles') as batch_op:
        batch_op.add_column(
            sa.Column('external_type', models.external_record.ExternalType(), server_default='', nullable=False)
        )
        batch_op.alter_column('__uniq_key_string__', existing_type=sa.CHAR(length=32), nullable=True)
        if is_mysql:
            batch_op.drop_constraint(u'user_roles_uq_1', type_='unique')

        batch_op.drop_constraint(u'user_roles_ibfk_1', type_='foreignkey')
        batch_op.drop_constraint(u'user_roles_ibfk_2', type_='foreignkey')
        batch_op.create_unique_constraint('user_roles_uq_1', ['secret_uuid', 'bundle_uuid', 'role_id', 'external_type', 'uid', 'staff_id', 'abc_id', 'abc_scope_id'])

    op.drop_index('idx_user_roles_abc_id', table_name='user_roles')
    op.create_index('idx_user_roles_abc_id_scope', 'user_roles', ['abc_id', 'abc_scope_id', 'external_type'], unique=False)

    op.drop_index('idx_user_roles_staff_id', table_name='user_roles')
    op.create_index('idx_user_roles_staff_id_type', 'user_roles', ['staff_id', 'external_type'], unique=False)

    if is_mysql:
        op.execute('BEGIN')

    op.execute(textwrap.dedent(
        '''
        UPDATE user_roles
           SET external_type =
                 CASE
                     WHEN abc_id IS NOT NULL THEN 'abc'
                     WHEN staff_id IS NOT NULL THEN 'staff'
                     WHEN uid IS NOT NULL THEN 'user'
                 END
        '''
    ))

    # В базе лежат записи только для development-скоупа. Обновляем скоуп в таблицах
    op.execute(textwrap.dedent(
        '''
           UPDATE external_records
              SET external_scope_id = '{}'
            WHERE external_type = 'abc'
                  AND external_scope_id = 0
        '''.format(ABC_DEVELOPMENT_SCOPE_ID)
    ))

    op.execute(textwrap.dedent(
        '''
           UPDATE user_roles
              SET abc_scope_id = '{}'
            WHERE external_type = 'abc'
                  AND (abc_scope_id = 0 OR abc_scope_id IS NULL)
        '''.format(ABC_DEVELOPMENT_SCOPE_ID)
    ))

    if is_mysql:
        op.execute('COMMIT')


    if is_mysql:
        op.execute('BEGIN')

    op.execute(textwrap.dedent(
        '''
        UPDATE user_roles
           SET staff_id = COALESCE(staff_id, 0),
               abc_id = COALESCE(abc_id, 0),
               abc_scope_id = COALESCE(abc_scope_id, {}),
               uid = COALESCE(uid, 0),
               secret_uuid = COALESCE(secret_uuid, ''),
               bundle_uuid = COALESCE(bundle_uuid, '')
        '''.format(ABC_DEVELOPMENT_SCOPE_ID)
    ))

    if is_mysql:
        op.execute('COMMIT')


def downgrade():
    is_mysql = (op.get_bind().dialect.name == 'mysql')

    op.drop_index('idx_user_roles_staff_id_type', table_name='user_roles')
    op.create_index('idx_user_roles_staff_id', 'user_roles', ['staff_id'], unique=False)

    op.create_index('idx_user_roles_abc_id', 'user_roles', ['abc_id'], unique=False)
    op.drop_index('idx_user_roles_abc_id_scope', table_name='user_roles')

    with op.batch_alter_table('user_roles') as batch_op:
        batch_op.alter_column('__uniq_key_string__', existing_type=sa.CHAR(length=32), nullable=False)

    if is_mysql:
        op.execute('BEGIN')

    op.execute(textwrap.dedent(
        '''
        UPDATE user_roles
           SET staff_id = NULLIF(staff_id, 0),
               abc_id = NULLIF(abc_id, 0),
               uid = NULLIF(uid, 0),
               secret_uuid = NULLIF(secret_uuid, ''),
               bundle_uuid = NULLIF(bundle_uuid, '')
        '''
    ))

    if is_mysql:
        op.execute(textwrap.dedent(
            '''
            UPDATE user_roles
               SET __uniq_key_string__ = MD5(CONCAT('abc_id=', abc_id, ',secret_uuid=sec-', secret_uuid))
             WHERE abc_id is not null
            '''
        ))

    if is_mysql:
        op.execute('COMMIT')

    with op.batch_alter_table('user_roles') as batch_op:
        if is_mysql:
            batch_op.drop_constraint(u'user_roles_uq_1', type_='unique')
        batch_op.create_unique_constraint('user_roles_uq_1', ['role_id', '__uniq_key_string__'])
        batch_op.drop_column('external_type')
        batch_op.drop_column('abc_scope_id')
        batch_op.create_foreign_key('user_roles_ibfk_1', 'secrets', ['secret_uuid'], ['uuid'])
        batch_op.create_foreign_key('user_roles_ibfk_2', 'bundles', ['bundle_uuid'], ['uuid'])

    with op.batch_alter_table('user_info') as batch_op:
        batch_op.drop_column('state')

    with op.batch_alter_table('staff_department_info') as batch_op:
        batch_op.drop_column('state')

    with op.batch_alter_table('abc_department_info') as batch_op:
        batch_op.drop_column('state')

    op.execute(textwrap.dedent(
        '''
        DELETE FROM external_records WHERE external_scope_id <> '{}'
        '''.format(ABC_DEVELOPMENT_SCOPE_ID)
    ))

    op.drop_index('idx_external_records_services', 'external_records')
    with op.batch_alter_table('external_records') as batch_op:
        if is_mysql:
            batch_op.drop_constraint(None, type_='primary')
        batch_op.create_primary_key(
            None if is_mysql else 'primary_key',
            ('external_type', 'external_id', 'uid'),
        )
        batch_op.drop_column('external_scope_id')

    op.drop_table('abc_departments_to_scopes')
    op.drop_table('abc_scopes')
