# -*- coding: utf-8 -*-

from passport.backend.vault.api.db import get_db
from passport.backend.vault.api.errors import (
    AbcRoleNotFoundError,
    AbcScopeNotFoundError,
    NonexistentEntityError,
)
from sqlalchemy import (
    or_,
    Table,
    UniqueConstraint,
)

from .base import (
    BaseModel,
    ExternalRecordState,
    MagicBigInteger,
    MagicInteger,
    Timestamp,
)


db = get_db()


class AbcDepartmentInfo(BaseModel):
    __tablename__ = 'abc_department_info'
    __repr_attrs__ = ['unique_name', 'state']

    default_serialization_columns = ['id', 'unique_name', 'display_name']
    default_serialization_pycolumns = ['state_name']

    id = db.Column(MagicBigInteger, primary_key=True, autoincrement=False)
    unique_name = db.Column(db.String(255))
    display_name = db.Column(db.String(255))

    state = db.Column(db.Integer, nullable=False, default=ExternalRecordState.normal.value, server_default='0')

    def state_name(self):
        return ExternalRecordState(self.state).name if self.state != ExternalRecordState.normal.value else None

    created_at = db.Column(Timestamp(current_timestamp=True), nullable=False)

    __table_args__ = (
        UniqueConstraint('unique_name', name='ix_abc_deparment_info_unique_name'),
    )

    scopes = db.relationship(
        'AbcScope',
        secondary=lambda: abc_departments_scopes_table,
        lazy='selectin',
        order_by='asc(AbcScope.unique_name)',
    )

    roles = db.relationship(
        'AbcRole',
        secondary=lambda: abc_departments_roles_table,
        lazy='selectin',
        order_by='asc(AbcRole.id)',
    )

    def scopes_ids_list(self):
        return tuple(sorted(map(lambda x: x.id, self.scopes)))

    def roles_ids_list(self):
        return tuple(sorted(map(lambda x: x.id, self.roles)))

    @staticmethod
    def get_by_unique_name(unique_name):
        abc_department = AbcDepartmentInfo.query.filter(AbcDepartmentInfo.unique_name == unique_name).first()
        if not abc_department:
            raise NonexistentEntityError(AbcDepartmentInfo, unique_name)
        return abc_department

    @staticmethod
    def get_by_ids(ids):
        return AbcDepartmentInfo.query.filter(
            AbcDepartmentInfo.id.in_(ids),
        ).all()

    @staticmethod
    def get_by_query(query, limit=None):
        like_query = u'%{query}%'.format(query=query)

        or_args = [
            AbcDepartmentInfo.unique_name.ilike(like_query),
            AbcDepartmentInfo.display_name.ilike(like_query),
        ]

        try:
            int_query = int(query)
        except (TypeError, ValueError):
            pass
        else:
            or_args.append(
                AbcDepartmentInfo.id == int_query,
            )

        orm_query = AbcDepartmentInfo.query.filter(
            AbcDepartmentInfo.state == ExternalRecordState.normal.value,
            or_(*or_args),
        ).order_by(
            AbcDepartmentInfo.display_name,
        ).limit(limit)

        return orm_query.all()

    @staticmethod
    def get_abc_service(abc_id=None, abc_slug=None, abc_scope_slug=None):
        filters = (AbcScope.unique_name == abc_scope_slug)
        if abc_slug is not None:
            filters = filters & (AbcDepartmentInfo.unique_name == abc_slug)
        else:
            filters = filters & (AbcDepartmentInfo.id == abc_id)
        return AbcDepartmentInfo.filter(filters).one()

    def __hash__(self):
        return hash((self.id, self.unique_name, self.display_name, self.scopes_ids_list(), self.roles_ids_list()))

    def __eq__(self, other):
        return (
            (self.id, self.unique_name, self.display_name, self.scopes_ids_list(), self.roles_ids_list()) ==
            (other.id, other.unique_name, other.display_name, other.scopes_ids_list(), other.roles_ids_list())
        )


class AbcScope(BaseModel):
    __tablename__ = 'abc_scopes'
    __repr_attrs__ = ['id', 'unique_name']
    default_serialization_columns = ['unique_name', 'display_name']

    id = db.Column(MagicInteger, primary_key=True, autoincrement=False)
    unique_name = db.Column(db.String(255), nullable=False)
    display_name = db.Column(db.String(255), nullable=False)

    created_at = db.Column(Timestamp(current_timestamp=True), nullable=False)

    __table_args__ = (
        UniqueConstraint('unique_name', name='ix_abc_scopes_unique_name'),
    )

    def __eq__(self, other):
        return (self.id, self.unique_name) == (other.id, other.unique_name)

    @classmethod
    def get_by_name(cls, name):
        scope = cls.query.filter(cls.unique_name == name).one_or_none()
        if scope is None:
            raise AbcScopeNotFoundError(None, name)
        return scope


abc_departments_scopes_table = Table(
    'abc_departments_to_scopes',
    BaseModel.metadata,
    db.Column(
        'abc_department_info',
        MagicBigInteger,
        db.ForeignKey('abc_department_info.id', name='abc_departments_to_scopes_ibfk_1'),
        primary_key=True,
    ),
    db.Column(
        'abc_scopes',
        MagicInteger,
        db.ForeignKey('abc_scopes.id', name='abc_departments_to_scopes_ibfk_2'),
        primary_key=True,
    ),
)


class AbcRole(BaseModel):
    __tablename__ = 'abc_roles'
    __repr_attrs__ = ['id', 'english_name']
    default_serialization_columns = ['id', 'display_name']

    id = db.Column(MagicInteger, primary_key=True, autoincrement=False)
    display_name = db.Column(db.String(255), nullable=False)
    english_name = db.Column(db.String(255), nullable=False)

    created_at = db.Column(Timestamp(current_timestamp=True), nullable=False)

    def __eq__(self, other):
        return self.id == other.id

    @classmethod
    def get_by_id(cls, abc_role_id):
        role = cls.query.filter(cls.id == abc_role_id).one_or_none()
        if role is None:
            raise AbcRoleNotFoundError(None, abc_role_id)
        return role


abc_departments_roles_table = Table(
    'abc_departments_to_roles',
    BaseModel.metadata,
    db.Column(
        'abc_department_info',
        MagicBigInteger,
        db.ForeignKey('abc_department_info.id', name='abc_departments_to_roles_ibfk_1'),
        primary_key=True,
    ),
    db.Column(
        'abc_roles',
        MagicInteger,
        db.ForeignKey('abc_roles.id', name='abc_departments_to_roles_ibfk_2'),
        primary_key=True,
    ),
)
