# -*- coding: utf-8 -*-

import time

from passport.backend.vault.api.db import get_db
from sqlalchemy import Index

from .base import (
    BaseModel,
    State,
    UpdatableMixin,
    UUIDType,
)
from .user_info import CreatorMixin


db = get_db()


class BundleUUIDType(UUIDType):
    prefix = 'bun'


class Bundle(BaseModel, UpdatableMixin, CreatorMixin):
    __tablename__ = 'bundles'
    __repr_attrs__ = ['name']

    default_serialization_columns = [
        'uuid', 'name', 'comment', 'created_at', 'created_by',
        'updated_at', 'updated_by', 'bundle_roles', 'bundle_versions',
    ]

    default_serialization_pycolumns = [
        'creator_login',
    ]

    uuid = db.Column(BundleUUIDType, primary_key=True, default=lambda: BundleUUIDType.create_ulid())
    name = db.Column(db.String(255))
    comment = db.Column(db.String(1023), nullable=True)

    state = db.Column(db.Integer, nullable=False, default=State.normal.value, server_default='0')

    @property
    def transitive_state(self):
        return self.state

    def state_name(self):
        return State(self.transitive_state).name

    creator_user_info = CreatorMixin.creator_relationship('Bundle')

    __table_args__ = (
        Index('idx_bundles_created_at', 'created_at'),
        Index('idx_bundles_created_by', 'created_by'),
        Index('idx_bundles_updated_at', 'updated_at'),
        Index('idx_bundles_updated_by', 'updated_by'),
    )

    bundle_versions = db.relationship(
        'BundleVersion',
        lazy='dynamic',
        order_by='desc(BundleVersion.created_at)',
        primaryjoin='and_(Bundle.uuid == foreign(BundleVersion.bundle_uuid), '
                    'BundleVersion.state == %s)' % State.normal.value,
    )

    bundle_roles = db.relationship(
        'UserRole',
        primaryjoin='Bundle.uuid == foreign(UserRole.bundle_uuid)',
        backref=db.backref('bundle', lazy='joined'),
        lazy='select',
        innerjoin=True,
        order_by='UserRole.role_id, UserRole.abc_id, UserRole.staff_id, UserRole.uid',
    )

    @staticmethod
    def create_bundle(created_by, name, comment=None):
        current_time = time.time()
        bundle_uuid = BundleUUIDType.create_ulid()
        bundle = Bundle(
            uuid=bundle_uuid,
            name=name,
            comment=comment,
            created_at=current_time,
            created_by=created_by,
            updated_at=current_time,
            updated_by=created_by,
        )

        return bundle

    @staticmethod
    def update_bundle(updated_by, bundle, name=None, comment=None, state=None):
        if name:
            bundle.name = name
        if comment is not None:
            bundle.comment = comment
        if state is not None:
            bundle.state = State[state].value

        bundle.touch(updated_by=updated_by)

        return bundle

    @staticmethod
    def get_bundle_with_versions(bundle_uuid, page, page_size):
        bundle = Bundle.get_by_id(bundle_uuid)
        bundle_versions = bundle.bundle_versions.limit(page_size).offset(page * page_size)
        return bundle, list(bundle_versions)
