# -*- coding: utf-8 -*-

import time

from passport.backend.vault.api.db import get_db
from sqlalchemy import (
    ForeignKeyConstraint,
    Index,
    Table,
)

from .base import (
    BaseModel,
    State,
    UUIDType,
)
from .bundle import (
    Bundle,
    BundleUUIDType,
)
from .secret_version import SecretVersionUUIDType
from .user_info import CreatorMixin


db = get_db()


class BundleVersionUUIDType(UUIDType):
    prefix = 'bve'


bundle_version_to_secret_version = Table(
    'bundle_version_to_secret_version',
    BaseModel.metadata,
    db.Column(
        'bundle_version',
        BundleVersionUUIDType,
        db.ForeignKey('bundle_versions.version', name='bundle_version_to_secret_version_ibfk_1'),
        primary_key=True,
    ),
    db.Column(
        'secret_version',
        SecretVersionUUIDType,
        db.ForeignKey('secret_versions.version', name='bundle_version_to_secret_version_ibfk_2'),
        primary_key=True,
    ),
)


class BundleVersion(BaseModel, CreatorMixin):
    __tablename__ = 'bundle_versions'
    __repr_attrs__ = ['bundle_uuid']

    max_serialization_depth = 5
    default_serialization_columns = [
        'version', 'created_at', 'created_by', 'comment', 'secret_versions',
    ]

    default_serialization_pycolumns = [
        'creator_login',
    ]

    version = db.Column(BundleVersionUUIDType, primary_key=True, default=lambda: BundleVersionUUIDType.create_ulid())
    bundle_uuid = db.Column(BundleUUIDType, nullable=False)
    bundle = db.relationship(
        'Bundle',
        lazy='joined',
        uselist=False,
    )

    creator_user_info = CreatorMixin.creator_relationship('BundleVersion')

    comment = db.Column(db.String(1023), nullable=True)

    state = db.Column(db.Integer, nullable=False, default=State.normal.value, server_default='0')

    @property
    def transitive_state(self):
        return max(self.state, self.bundle.state)

    def state_name(self):
        return State(self.transitive_state).name

    # TODO: Сделать ttl для версий бандлов.
    #       Пока оставили заглушки для совместимости с версиями секретов.

    expired_at = None

    @staticmethod
    def check_expired(version):  # pragma: no cover
        pass

    @property
    def expired(self):  # pragma: no cover
        pass

    def set_ttl(self, ttl=None):
        pass

    def touch(self, *args, **kwargs):
        pass

    secret_versions = db.relationship(
        'SecretVersion',
        secondary='bundle_version_to_secret_version',
        secondaryjoin=(
            'and_(bundle_version_to_secret_version.c.secret_version == foreign(SecretVersion.version), '
            'SecretVersion.state == %s)' % State.normal.value
        ),
        lazy='select',
    )

    __table_args__ = (
        ForeignKeyConstraint(['bundle_uuid'], [Bundle.uuid], name='bundle_versions_ibfk_1'),
        Index('idx_bundle_versions_bundle_uuid', 'bundle_uuid'),
        Index('idx_bundle_versions_created_at', 'created_at'),
        Index('idx_bundle_versions_created_by', 'created_by'),
    )

    @staticmethod
    def create_bundle_version(bundle_uuid, created_by, comment=None):
        current_time = time.time()
        bundle = BundleVersion(
            version=BundleVersionUUIDType.create_ulid(),
            bundle_uuid=bundle_uuid,
            comment=comment,
            created_at=current_time,
            created_by=created_by,
        )

        return bundle
