# -*- coding: utf-8 -*-

import time

from passport.backend.vault.api.db import get_db
from passport.backend.vault.api.errors import InvalidDelegationTokenError
from passport.backend.vault.api.models.base import (
    BaseModel,
    MagicBigInteger,
    Timestamp,
    TokenState,
    UUIDType,
)
from passport.backend.vault.api.models.secret import (
    Secret,
    SecretUUIDType,
)
from passport.backend.vault.api.models.user_info import CreatorMixin
from passport.backend.vault.api.utils import secrets
from sqlalchemy import (
    ForeignKeyConstraint,
    Index,
    UniqueConstraint,
)


db = get_db()


class DelegationTokenUUIDType(UUIDType):
    prefix = 'tid'


class DelegationToken(BaseModel, CreatorMixin):
    __tablename__ = 'delegation_tokens'
    __repr_attrs__ = ['secret_uuid']

    default_serialization_columns = ['secret_uuid', 'tvm_client_id', 'tvm_app', 'signature', 'comment', 'created_by', 'created_at', 'revoked_at']
    default_serialization_pycolumns = ['token_uuid', 'creator_login', 'revocation_login', 'state_name']

    id = db.Column(DelegationTokenUUIDType, primary_key=True, default=lambda: DelegationTokenUUIDType.create_ulid())

    def token_uuid(self):
        return self.id

    hashed_token = db.Column(db.String(128))

    secret_uuid = db.Column(SecretUUIDType, nullable=False)
    secret = db.relationship(
        'Secret',
        primaryjoin='DelegationToken.secret_uuid == foreign(Secret.uuid)',
        lazy='dynamic',
        viewonly=True,
    )

    tvm_client_id = db.Column(MagicBigInteger, nullable=True)
    tvm_app = db.relationship(
        'TvmAppInfo',
        primaryjoin='DelegationToken.tvm_client_id == foreign(TvmAppInfo.tvm_client_id)',
        lazy='subquery',
        uselist=False,
        viewonly=True,
    )

    signature = db.Column(db.String(255))
    comment = db.Column(db.String(1023), nullable=True)

    creator_user_info = CreatorMixin.creator_relationship('DelegationToken')

    state = db.Column(db.Integer, nullable=False, default=TokenState.normal.value, server_default='0')

    def state_name(self):
        return TokenState(self.state).name

    revoked_by = db.Column(MagicBigInteger, nullable=True)
    revoked_at = db.Column(Timestamp(), nullable=True)
    revocation_user_info = db.relationship(
        'UserInfo',
        primaryjoin='DelegationToken.revoked_by == foreign(UserInfo.uid)',
        lazy='subquery',
        uselist=False,
        viewonly=True,
    )

    def revocation_login(self):
        return self.revocation_user_info.login if self.revocation_user_info else None

    __table_args__ = (
        ForeignKeyConstraint(['secret_uuid'], [Secret.uuid], name='delegation_tokens_ibfk_1'),
        UniqueConstraint('hashed_token'),
        Index('idx_delegation_tokens_secret_uuid', 'secret_uuid'),
        Index('idx_delegation_tokens_created_at', 'created_at'),
        Index('idx_delegation_tokens_created_by', 'created_by'),
    )

    def revoke(self, revoked_by=None):
        self.state = TokenState.revoked.value
        self.revoked_at = time.time()
        self.revoked_by = revoked_by

    def restore(self):
        self.state = TokenState.normal.value
        self.revoked_at = None
        self.revoked_by = None

    @classmethod
    def build_token(cls):
        return secrets.sign_string(
            secrets.token_urlsafe(cls.config['application']['delegation_token_length']),
        )

    @classmethod
    def hash_token(cls, token):
        return secrets.default_hash(secrets.parse_signed_string(token, verify=False))

    @classmethod
    def mask_token(cls, token):
        min_masked_length = 20
        length = len(token)
        if length <= min_masked_length:
            return '*' * length
        prefix_len = min_masked_length // 2 - 1
        return token[:prefix_len] + '*' * (length - prefix_len * 2) + token[-1 * prefix_len:]

    @classmethod
    def validate_token(cls, token):
        try:
            secrets.parse_signed_string(token)
        except secrets.InvalidSignedString:
            raise InvalidDelegationTokenError(token=cls.mask_token(token))
        return True

    @classmethod
    def create_token(cls, created_by, secret_uuid, tvm_client_id=None, signature=None, comment=None):
        token = cls.build_token()
        current_time = time.time()
        delegation_token = DelegationToken(
            hashed_token=DelegationToken.hash_token(token),
            secret_uuid=secret_uuid,
            tvm_client_id=tvm_client_id,
            signature=signature,
            comment=comment,
            created_at=current_time,
            created_by=created_by,
        )
        return token, delegation_token

    @classmethod
    def get_count(cls, type_=None, **kwargs):
        conditions = None

        if type_:
            conditions = (DelegationToken.state == TokenState[type_].value)

        return DelegationToken.query.filter(conditions).count()
