# -*- coding: utf-8 -*-
import logging

from passport.backend.vault.api.db import (
    chunked_delete,
    chunked_merge,
    get_db,
)
from passport.backend.vault.api.models.abc_department_info import AbcDepartmentInfo
from passport.backend.vault.api.models.base import (
    BaseModel,
    ExternalRecordState,
    MagicBigInteger,
    MagicInteger,
    Timestamp,
)
from passport.backend.vault.api.models.staff_department_info import StaffDepartmentInfo
from passport.backend.vault.api.models.user_info import UserInfo
from sqlalchemy import (
    Index,
    or_,
    PrimaryKeyConstraint,
)
from sqlalchemy.orm import undefer
from sqlalchemy.types import TypeDecorator


db = get_db()


class ExternalType(TypeDecorator):
    available_types = [
        'abc',
        'staff',
        'user',
    ]

    impl = db.VARCHAR(100)

    def process_bind_param(self, value, dialect):
        value = value.strip().lower()
        if value not in self.available_types:
            raise ValueError('"%s" is an unknown external type' % value)
        return value

    def process_result_value(self, value, dialect):
        value = value.strip().lower()
        if value not in self.available_types:
            raise ValueError('"%s" is an unknown external type' % value)
        return value


def get_inactivated_records_by_id(cached_records, new_records):
    new_ids = set(map(lambda x: x.identity, new_records))
    result = filter(
        lambda x: x.identity not in new_ids and x.state == ExternalRecordState.normal.value,
        cached_records,
    )
    return result


def get_restored_records(cached_records, new_records):
    cached = dict(map(lambda x: (x.identity, x), cached_records))
    result = filter(
        lambda x: x.identity in cached and cached[x.identity].state == ExternalRecordState.inactive.value,
        cached_records,
    )
    return result


class ExternalRecord(BaseModel):
    __tablename__ = 'external_records'

    external_type = db.Column(ExternalType)
    external_id = db.Column(MagicBigInteger)
    external_scope_id = db.Column(MagicInteger, nullable=False, default=0, server_default='0')
    external_role_id = db.Column(MagicInteger, nullable=False, default=0, server_default='0')

    uid = db.Column(MagicBigInteger)
    login = db.Column(db.String(255))

    created_at = db.Column(Timestamp(current_timestamp=True), nullable=False)

    __table_args__ = (
        PrimaryKeyConstraint('uid', 'external_type', 'external_id', 'external_scope_id', 'external_role_id'),
        Index('idx_external_records_uid', 'uid'),
        Index('idx_external_records_login', 'login'),
        Index('idx_external_records_services', 'external_type', 'external_id',  'external_role_id'),
        Index('idx_external_records_services_roles', 'external_type', 'external_id', 'external_scope_id', 'external_role_id'),
    )

    @staticmethod
    def get_groups(external_record_type, uid):
        filters = (ExternalRecord.external_type == external_record_type)
        if uid is not None:
            filters = filters & (ExternalRecord.uid == uid)
        if external_record_type == 'abc':
            filters = filters & or_(
                ExternalRecord.external_scope_id > 0,
                ExternalRecord.external_role_id > 0,
            )
        external_records = ExternalRecord.query.with_entities(
            ExternalRecord.external_id,
            ExternalRecord.external_scope_id,
            ExternalRecord.external_role_id,
        ).filter(filters).all()
        return list(map(
            lambda x: (int(x.external_id), int(x.external_scope_id), int(x.external_role_id)),
            external_records,
        ))

    @staticmethod
    def get_abc_groups(uid):
        return ExternalRecord.get_groups('abc', uid)

    @staticmethod
    def get_staff_groups(uid):
        return map(
            lambda x: x[0],
            ExternalRecord.get_groups('staff', uid),
        )

    def __hash__(self):
        return hash((self.external_type, self.external_id, self.uid, self.external_scope_id, self.external_role_id))

    def __eq__(self, other):
        return (
            (self.external_type, self.external_id, self.uid, self.external_scope_id, self.external_role_id) ==
            (other.external_type, other.external_id, other.uid, other.external_scope_id, other.external_role_id)
        )

    @classmethod
    def insert_abc(cls, abc_persons, abc_scopes, abc_services_scopes, abc_departments, abc_roles, abc_services_roles):
        logging.getLogger('info_logger').info('ABC: inserting...')
        cached_external_records = set(ExternalRecord.query.filter(ExternalRecord.external_type == 'abc').all())
        cached_abc_department_infos = set(AbcDepartmentInfo.query.all())
        actual_external_records = set()
        actual_abc_department_infos = set()

        for uid in abc_persons:
            for abc_id, abc_scope_id in abc_persons[uid]['group_ids']:
                actual_external_records.add(
                    ExternalRecord(
                        external_type='abc',
                        external_id=abc_id,
                        external_scope_id=abc_scope_id,
                        external_role_id=0,
                        uid=uid,
                        login=abc_persons[uid]['login'],
                    ),
                )
            for abc_id, abc_role_id in abc_persons[uid]['roles_ids']:
                actual_external_records.add(
                    ExternalRecord(
                        external_type='abc',
                        external_id=abc_id,
                        external_scope_id=0,
                        external_role_id=abc_role_id,
                        uid=uid,
                        login=abc_persons[uid]['login'],
                    ),
                )

        for abc_service_id in abc_departments:
            abc_service = AbcDepartmentInfo(
                id=abc_service_id,
                unique_name=abc_departments[abc_service_id]['slug'],
                display_name=abc_departments[abc_service_id]['name'],
                state=ExternalRecordState.normal.value,
            )
            for abc_scope_id in abc_services_scopes.get(abc_service_id, list()):
                abc_service.scopes.append(abc_scopes[abc_scope_id])
            for abc_role_id in abc_services_roles.get(abc_service_id, list()):
                abc_service.roles.append(abc_roles[abc_role_id])
            actual_abc_department_infos.add(abc_service)

        new_external_records = list(actual_external_records - cached_external_records)
        old_external_records = list(cached_external_records - actual_external_records)

        restored_department_infos = get_restored_records(
            cached_abc_department_infos,
            actual_abc_department_infos,
        )
        new_department_infos = list(
            actual_abc_department_infos - (cached_abc_department_infos - set(restored_department_infos))
        )

        chunked_delete(
            old_external_records,
            cls.config['abc']['insert_chunk_size'],
            'ABC: removed chunk of %s external records',
        )

        chunked_merge(
            new_external_records,
            cls.config['abc']['insert_chunk_size'],
            'ABC: upserted chunk of %s external records',
        )
        chunked_merge(
            new_department_infos,
            cls.config['abc']['insert_chunk_size'],
            'ABC: upserted chunk of %s services',
        )

        old_department_infos = get_inactivated_records_by_id(
            cached_abc_department_infos,
            actual_abc_department_infos,
        )
        if old_department_infos:
            for d in old_department_infos:
                d.state = ExternalRecordState.inactive.value
            chunked_merge(
                old_department_infos,
                cls.config['abc']['insert_chunk_size'],
                'ABC: deactivated chunk of %s services',
            )

        logging.getLogger('info_logger').info(
            'ABC: upserted {new_external_records} new external records, '
            'removed {old_external_records} old external records, '
            'deactivated {old_department_infos} old services, '
            'upserted {new_department_infos} services '.format(
                new_external_records=len(new_external_records),
                old_external_records=len(old_external_records),
                new_department_infos=len(new_department_infos),
                old_department_infos=len(old_department_infos),
            ),
        )
        return dict(
            new_external_records=len(new_external_records),
            old_external_records=len(old_external_records),
            new_department_infos=len(new_department_infos),
            old_department_infos=len(old_department_infos),
        )

    @classmethod
    def insert_staff(cls, staff_persons, staff_departments):
        logging.getLogger('info_logger').info('Staff: inserting...')
        cached_external_records = set(ExternalRecord.query.filter(ExternalRecord.external_type == 'staff').all())
        cached_user_infos = set(UserInfo.query.options(undefer('*')).all())
        cached_staff_department_infos = set(StaffDepartmentInfo.query.all())
        actual_external_records = set()
        actual_user_infos = set()
        actual_staff_department_infos = set()

        for staff_department_id in staff_departments:
            actual_staff_department_infos.add(
                StaffDepartmentInfo(
                    id=staff_department_id,
                    unique_name=staff_departments[staff_department_id]['url'],
                    display_name=staff_departments[staff_department_id]['name'],
                    state=ExternalRecordState.inactive.value if staff_departments[staff_department_id]['is_deleted'] else ExternalRecordState.normal.value,
                ),
            )

        for uid in staff_persons:
            for staff_id in staff_persons[uid]['group_ids']:
                if staff_persons[uid]['_disabled']:
                    continue
                actual_external_records.add(
                    ExternalRecord(
                        external_type='staff',
                        external_id=staff_id,
                        external_scope_id=0,
                        external_role_id=0,
                        uid=uid,
                        login=staff_persons[uid]['login'],
                    )
                )
            actual_user_infos.add(
                UserInfo(
                    uid=uid,
                    login=staff_persons[uid]['login'],
                    keys=staff_persons[uid]['keys'],
                    first_name=staff_persons[uid]['first_name'],
                    last_name=staff_persons[uid]['last_name'],
                    staff_id=staff_persons[uid]['staff_id'],
                    state=ExternalRecordState.inactive.value if staff_persons[uid]['_disabled'] else ExternalRecordState.normal.value,
                )
            )

        new_external_records = list(actual_external_records - cached_external_records)
        old_external_records = list(cached_external_records - actual_external_records)

        new_user_infos = list(actual_user_infos - cached_user_infos)
        new_department_infos = list(actual_staff_department_infos - cached_staff_department_infos)

        removed_external_records = (len(old_external_records) != len(cached_external_records))
        if removed_external_records:
            # Простая защита. Не даем снести все external_records
            chunked_delete(
                old_external_records,
                cls.config['staff']['insert_chunk_size'],
                'Staff: removed chunk of %s external records',
            )

        chunked_merge(
            new_external_records,
            cls.config['staff']['insert_chunk_size'],
            'Staff: upserted chunk of %s external records',
        )

        chunked_merge(
            new_user_infos,
            cls.config['staff']['insert_chunk_size'],
            'Staff: upserted chunk of %s users',
        )
        chunked_merge(
            new_department_infos,
            cls.config['staff']['insert_chunk_size'],
            'Staff: upserted chunk of %s departments',
        )

        logging.getLogger('info_logger').info(
            'Staff: upserted {new_external_records} new external records, '
            'removed {old_external_records} old external records, '
            'upserted {new_user_infos} users, '
            'upserted {new_department_infos} departments '.format(
                new_external_records=len(new_external_records),
                old_external_records=len(old_external_records) if removed_external_records else 0,
                new_user_infos=len(new_user_infos),
                new_department_infos=len(new_department_infos),
            ),
        )
        return dict(
            new_external_records=len(new_external_records),
            old_external_records=len(old_external_records) if removed_external_records else 0,
            new_user_infos=len(new_user_infos),
            new_department_infos=len(new_department_infos),
        )
