# -*- coding: utf-8 -*-

import time

from passport.backend.vault.api.db import get_db
from passport.backend.vault.api.models.base import (
    BaseModel,
    State,
    TokenState,
    UpdatableMixin,
    UUIDType,
)
from passport.backend.vault.api.models.tags import (
    Tag,
    TagEntity,
)
from passport.backend.vault.api.models.user_info import CreatorMixin
from sqlalchemy import Index
from sqlalchemy.ext.associationproxy import association_proxy


db = get_db()


class SecretUUIDType(UUIDType):
    prefix = 'sec'


class Secret(BaseModel, UpdatableMixin, CreatorMixin):
    __tablename__ = 'secrets'
    __repr_attrs__ = ['name']

    default_serialization_columns = [
        'uuid', 'name', 'comment', 'created_at', 'created_by',
        'updated_at', 'updated_by', 'secret_roles', 'secret_versions',
    ]
    default_serialization_pycolumns = [
        'tags', 'creator_login',
    ]

    uuid = db.Column(SecretUUIDType, primary_key=True, default=lambda: SecretUUIDType.create_ulid())
    name = db.Column(db.String(255))
    comment = db.Column(db.String(1023), nullable=True)

    creator_user_info = CreatorMixin.creator_relationship('Secret')

    state = db.Column(db.Integer, nullable=False, default=State.normal.value, server_default='0')

    @property
    def transitive_state(self):
        return self.state

    def state_name(self):
        return State(self.transitive_state).name

    __table_args__ = (
        Index('idx_secrets_created_at', 'created_at'),
        Index('idx_secrets_created_by', 'created_by'),
        Index('idx_secrets_updated_at', 'updated_at'),
        Index('idx_secrets_updated_by', 'updated_by'),
    )

    secret_versions = db.relationship(
        'SecretVersion',
        lazy='dynamic',
        order_by='desc(SecretVersion.version)',
        primaryjoin='and_(Secret.uuid == foreign(SecretVersion.secret_uuid), '
                    'SecretVersion.state == %s)' % State.normal.value,
    )

    def head_version(self):
        versions = filter(
            lambda x: (not x.hidden and not x.expired),
            self.secret_versions,
        )
        if len(versions):
            return versions[0]
        return None

    def versions_count(self):
        # ToDo: Circular import
        from passport.backend.vault.api.models import SecretVersion
        return self.secret_versions.filter(SecretVersion.unexpired_version_filter()).count()

    tokens = db.relationship(
        'DelegationToken',
        lazy='dynamic',
        order_by='asc(DelegationToken.id)',
        primaryjoin='and_(Secret.uuid == foreign(DelegationToken.secret_uuid), '
                    'DelegationToken.state == %s)' % TokenState.normal.value,
    )

    def tokens_count(self):
        return self.tokens.count()

    secret_roles = db.relationship(
        'UserRole',
        primaryjoin='Secret.uuid == foreign(UserRole.secret_uuid)',
        backref=db.backref('secret', lazy='joined'),
        lazy='select',
        order_by='UserRole.role_id, UserRole.abc_id, UserRole.staff_id, UserRole.uid, UserRole.abc_scope_id, UserRole.abc_role_id',
    )

    tags_links = TagEntity.tags_link_relationship(
        'secret',
        'Secret.uuid',
        lazy='select',
    )

    _tags = association_proxy('tags_links', 'tag')

    def tags(self):
        result = [tag.title for tag in self._tags] if self._tags else None
        return result or None

    @staticmethod
    def get_secret_with_versions(secret_uuid, page, page_size, filters=None):
        secret = Secret.get_by_id(secret_uuid)
        filters = filters if filters is not None else []
        secret_versions = secret.secret_versions.filter(*filters).limit(page_size).offset(page * page_size)
        return secret, list(secret_versions)

    @staticmethod
    def create_secret(created_by, name, comment=None, tags=None):
        current_time = time.time()
        secret_uuid = SecretUUIDType.create_ulid()
        secret = Secret(
            uuid=secret_uuid,
            name=name,
            comment=comment,
            created_at=current_time,
            created_by=created_by,
            updated_at=current_time,
            updated_by=created_by,
        )

        if tags is not None:
            tags_objects = Tag.create_tags(tags, created_by)
            secret.tags_links = TagEntity.build_links(
                tags_strings=tags_objects,
                old_links=[],
                entity_id=secret_uuid,
                entity_type='secret',
                created_by=created_by,
            )

        return secret

    @staticmethod
    def update_secret(updated_by, secret, name=None, comment=None, tags=None, state=None):
        if name:
            secret.name = name
        if comment is not None:
            secret.comment = comment
        if state is not None:
            secret.state = State[state].value

        if tags is not None:
            tags_objects = Tag.create_tags(tags, updated_by)
            secret.tags_links = TagEntity.build_links(
                tags_objects,
                secret.tags_links,
                secret.uuid,
                'secret',
                updated_by,
            )
        secret.touch(updated_by=updated_by)

        return secret
