# -*- coding: utf-8 -*-

from passport.backend.vault.api.db import get_db
from passport.backend.vault.api.errors import NonexistentEntityError
from sqlalchemy import (
    or_,
    UniqueConstraint,
)

from .base import (
    BaseModel,
    ExternalRecordState,
    MagicBigInteger,
    Timestamp,
)


db = get_db()


class StaffDepartmentInfo(BaseModel):
    __tablename__ = 'staff_department_info'
    __repr_attrs__ = ['unique_name', 'state']

    default_serialization_columns = [
        'id', 'unique_name', 'display_name',
    ]
    default_serialization_pycolumns = ['state_name']

    id = db.Column(MagicBigInteger, primary_key=True, autoincrement=False)
    unique_name = db.Column(db.String(255))
    display_name = db.Column(db.String(255))

    state = db.Column(db.Integer, nullable=False, default=ExternalRecordState.normal.value, server_default='0')

    def state_name(self):
        return ExternalRecordState(self.state).name if self.state != ExternalRecordState.normal.value else None

    created_at = db.Column(Timestamp(current_timestamp=True), nullable=False)

    __table_args__ = (
        UniqueConstraint('unique_name', name='ix_staff_deparment_info_unique_name'),
    )

    @staticmethod
    def get_by_unique_name(unique_name):
        staff_department = StaffDepartmentInfo.query.filter(
            StaffDepartmentInfo.unique_name == unique_name,
        ).first()
        if not staff_department:
            raise NonexistentEntityError(StaffDepartmentInfo, unique_name)
        return staff_department

    @staticmethod
    def get_by_ids(ids):
        return StaffDepartmentInfo.query.filter(StaffDepartmentInfo.id.in_(ids)).all()

    @staticmethod
    def get_by_query(query, limit=None):
        query = u'%{query}%'.format(query=query)
        return StaffDepartmentInfo.query.filter(
            StaffDepartmentInfo.state == ExternalRecordState.normal.value,
            or_(
                StaffDepartmentInfo.unique_name.ilike(query),
                StaffDepartmentInfo.display_name.ilike(query),
            ),
        ).order_by(
            StaffDepartmentInfo.display_name,
        ).limit(limit).all()

    def __hash__(self):
        return hash((self.id, self.unique_name, self.display_name, self.state))

    def __eq__(self, other):
        return (self.id, self.unique_name, self.display_name, self.state) == (other.id, other.unique_name, other.display_name, other.state)
