# coding: utf-8

import logging
import re
import warnings

from passport.backend.vault.api.db import get_db
from passport.backend.vault.api.models.base import (
    BaseModel,
    MagicBigInteger,
    MagicInteger,
    Timestamp,
    UUIDType,
)
from passport.backend.vault.api.utils import ulid
import six
from sqlalchemy import (
    and_,
    exc as sa_exc,
    ForeignKeyConstraint,
    func,
    Index,
    PrimaryKeyConstraint,
    types,
    UniqueConstraint,
)
from sqlalchemy.exc import IntegrityError


db = get_db()

MAX_TAG_LENGTH = 127
STRIP_DOUBLE_SPACES_RE = re.compile(r'\s+')

ENTITY_TYPES = {
    'secret': 1,
}


class TagTitle(types.TypeDecorator):
    impl = types.VARCHAR(MAX_TAG_LENGTH)

    def process_bind_param(self, value, dialect):
        value = Tag.normalize_tag_title(value)
        if not value:
            ValueError('Empty tag')  # pragma: no cover
        return value


class Tag(BaseModel):
    __tablename__ = 'tags'
    __repr_attrs__ = ['title']

    tag_id = db.Column(UUIDType, primary_key=True, default=lambda: ulid.create_ulid())
    title = db.Column(TagTitle, nullable=False)
    created_at = db.Column(Timestamp(current_timestamp=True), nullable=False)
    created_by = db.Column(MagicBigInteger, nullable=False)

    __table_args__ = (
        UniqueConstraint('title'),
    )

    @staticmethod
    def normalize_tag_title(title):
        title = title.replace(',', ' ')
        title = STRIP_DOUBLE_SPACES_RE.sub(' ', title)
        title = title.strip()
        return title

    @staticmethod
    def normalize_tags(tags_strings):
        if not tags_strings:
            return []
        uniq_tags = set()
        result = []
        for tag_title in tags_strings:
            tag_title = Tag.normalize_tag_title(tag_title)
            if tag_title != '' and tag_title.lower() not in uniq_tags:
                uniq_tags.add(tag_title.lower())
                result.append(tag_title)
        return result

    @staticmethod
    def find_tags(text, limit=None, offset=None):
        text = Tag.normalize_tag_title(text)

        if len(text) > 0:
            return Tag.query.filter(
                Tag.title.like(u'%{}%'.format(text)),
            ).order_by(
                Tag.title.asc(),
            ).offset(
                offset,
            ).limit(
                limit,
            ).all()

    @staticmethod
    def create_tags(tags_strings, created_by, created_at=None):
        tags_strings = Tag.normalize_tags(tags_strings)
        with warnings.catch_warnings():
            warnings.simplefilter("ignore", category=sa_exc.SAWarning)
            for tag_title in tags_strings:
                new_tag = Tag(
                    title=tag_title,
                    created_by=created_by,
                    created_at=created_at,
                )
                try:
                    with db.session.begin_nested():
                        db.session.merge(new_tag)
                except IntegrityError:
                    pass
                except Exception as e:  # pragma: no cover
                    logging.getLogger('exception_logger').exception(e, exc_info=True)
                    raise

        return db.session.query(Tag).filter(Tag.title.in_(tags_strings)).all()


class TagEntity(BaseModel):
    __tablename__ = 'tags_entities'

    tag_id = db.Column(UUIDType, nullable=False)
    entity_type_id = db.Column(MagicInteger, nullable=False)
    entity_id = db.Column(UUIDType(ignore_prefix=True), nullable=False)
    created_at = db.Column(Timestamp(current_timestamp=True), nullable=False)
    created_by = db.Column(MagicBigInteger, nullable=False)

    tag = db.relationship(
        'Tag',
        lazy='joined',
        innerjoin=True,
        backref=db.backref('entities', lazy='dynamic'),
        order_by='asc(Tag.title)',
    )

    __table_args__ = (
        PrimaryKeyConstraint('tag_id', 'entity_type_id', 'entity_id'),
        ForeignKeyConstraint(['tag_id'], [Tag.tag_id], name='tags_entities_ibfk_1'),
        Index('idx_tags_entities_entity_type_id_tag_id', 'entity_type_id', 'tag_id'),
    )

    @staticmethod
    def tags_link_relationship(entity_type, entity_ref, lazy='joined', uselist=True):
        return db.relationship(
            'TagEntity',
            primaryjoin='and_(TagEntity.entity_id == foreign({entity_ref}), '
                        'TagEntity.entity_type_id == \'{entity_type_id}\')'
                        .format(
                            entity_type_id=ENTITY_TYPES[entity_type],
                            entity_ref=entity_ref,
                        ),
            uselist=uselist,
            lazy=lazy,
        )

    @staticmethod
    def tag_entity_join_filters(entity_type, entity_column):
        return and_(
            TagEntity.entity_id == entity_column,
            TagEntity.entity_type_id == ENTITY_TYPES[entity_type],
        )

    @staticmethod
    def tags_exists_filter(entity_type, entity_column, tags):
        return TagEntity.query.filter(
            TagEntity.tag_id == Tag.tag_id,
            TagEntity.entity_type_id == ENTITY_TYPES[entity_type],
            Tag.title.in_(tags),
        ).group_by(
            TagEntity.entity_id,
        ).having(
            func.count(Tag.tag_id) == len(tags),
        ).exists().where(
            TagEntity.entity_id == entity_column,
        )

    @staticmethod
    def build_links(tags_strings, old_links, entity_id, entity_type, created_by, created_at=None):
        tags_dict = dict(map(lambda x: (x.tag_id, x), tags_strings))
        links_dict = dict(map(lambda x: (x.tag_id, x), old_links))

        # Оставляем линки, котрые не поменялись
        result = [links_dict[tag_id] for tag_id in (six.viewkeys(tags_dict) & six.viewkeys(links_dict))]

        # Удаляем старые линки
        for tag_id in (six.viewkeys(links_dict) - six.viewkeys(tags_dict)):
            db.session.delete(links_dict[tag_id])

        # Вставляем новые
        for tag_id in (six.viewkeys(tags_dict) - six.viewkeys(links_dict)):
            result.append(
                TagEntity(
                    tag=tags_dict[tag_id],
                    entity_id=entity_id,
                    entity_type_id=ENTITY_TYPES[entity_type],
                    created_by=created_by,
                    created_at=created_at,
                ),
            )

        return result
