# -*- coding: utf-8 -*-

import logging

from passport.backend.vault.api.db import (
    chunked_merge,
    get_db,
)
from passport.backend.vault.api.models.base import (
    BaseModel,
    MagicBigInteger,
    Timestamp,
)
from sqlalchemy import (
    Index,
    or_,
    UniqueConstraint,
)


db = get_db()


class TvmAppInfo(BaseModel):
    __tablename__ = 'tvm_apps_info'
    __repr_attrs__ = ['name', 'abc_state']

    default_serialization_columns = ['tvm_client_id', 'name', 'url', 'abc_state', 'abc_state_display_name', 'abc_department']

    tvm_client_id = db.Column(MagicBigInteger, primary_key=True, autoincrement=False)
    name = db.Column(db.String(255))
    url = db.Column(db.String(1024))

    abc_id = db.Column(MagicBigInteger, nullable=False)
    abc_resource_id = db.Column(MagicBigInteger, nullable=False)
    abc_state = db.Column(db.String(255))
    abc_state_display_name = db.Column(db.String(255))

    created_at = db.Column(Timestamp(current_timestamp=True), nullable=False)

    abc_department = db.relationship(
        'AbcDepartmentInfo',
        primaryjoin='TvmAppInfo.abc_id == foreign(AbcDepartmentInfo.id)',
        lazy='select',
        viewonly=True,
        uselist=False,
    )

    __table_args__ = (
        UniqueConstraint('abc_resource_id', name='idx_uq_tvm_apps_info_abc_resource_id'),
        Index('idx_tvm_apps_info_abc_id', 'abc_id'),
    )

    @staticmethod
    def get_by_query(query, abc_state=None, limit=None):
        query = u'%{query}%'.format(query=query)
        filters = or_(
            TvmAppInfo.name.ilike(query),
            TvmAppInfo.tvm_client_id.ilike(query),
        )
        if abc_state is not None:
            filters = filters & (TvmAppInfo.abc_state == abc_state)

        return TvmAppInfo.query.filter(
            filters,
        ).order_by(
            TvmAppInfo.name,
            TvmAppInfo.tvm_client_id,
        ).limit(
            limit,
        ).all()

    @classmethod
    def insert_tvm_apps(cls, tvm_apps):
        logging.getLogger('info_logger').info('TVM apps: inserting...')
        tvm_apps = map(
            lambda x: TvmAppInfo(
                abc_id=x['abc_id'],
                abc_resource_id=x['abc_resource_id'],
                abc_state=x['abc_state'],
                abc_state_display_name=x['abc_state_display_name'],
                name=x['name'],
                tvm_client_id=x['tvm_client_id'],
                url=x['url'],
            ),
            tvm_apps.values(),
        )
        chunked_merge(
            tvm_apps,
            cls.config['tvm_apps']['insert_chunk_size'],
            'TVM apps: upserted chunk of %s records',
        )
        return dict(
            tvm_apps=len(tvm_apps),
        )
