# coding: utf-8

from passport.backend.vault.api.db import get_db
from passport.backend.vault.api.errors import (
    TvmGrantExistsError,
    TvmGrantRequiredError,
)
from passport.backend.vault.api.models.base import (
    BaseModel,
    MagicBigInteger,
    Timestamp,
)
from sqlalchemy.exc import IntegrityError
from sqlalchemy.orm import joinedload


db = get_db()


class TvmGrants(BaseModel):
    __tablename__ = 'tvm_grants'
    __repr_attrs__ = ['tvm_client_id']

    default_serialization_columns = ['tvm_client_id', 'tvm_app', 'comment', 'created_at']

    tvm_client_id = db.Column(MagicBigInteger, primary_key=True, autoincrement=False)
    comment = db.Column(db.String(1023), nullable=True)
    created_at = db.Column(Timestamp(current_timestamp=True), nullable=False)

    tvm_app = db.relationship(
        'TvmAppInfo',
        primaryjoin='TvmGrants.tvm_client_id == foreign(TvmAppInfo.tvm_client_id)',
        lazy='select',
        uselist=False,
        viewonly=True,
    )

    @classmethod
    def fetch(cls, tvm_client_id):
        return cls.query.filter(
            cls.tvm_client_id == tvm_client_id
        ).options(
            joinedload('tvm_app'),
        ).one_or_none()

    @classmethod
    def list(cls):
        return cls.query.order_by(
            cls.tvm_client_id,
        ).options(
            joinedload('tvm_app'),
        ).all()

    @classmethod
    def grant(cls, tvm_client_id, comment=None):
        try:
            db.session.add(
                cls(
                    tvm_client_id=tvm_client_id,
                    comment=comment,
                ),
            )
            db.session.commit()
        except IntegrityError:
            raise TvmGrantExistsError()

    @classmethod
    def revoke(cls, tvm_client_id):
        cls.query.filter(
            cls.tvm_client_id == tvm_client_id,
        ).delete()
        db.session.commit()

    @classmethod
    def check(cls, tvm_client_id):
        grant = cls.query.filter(
            cls.tvm_client_id == tvm_client_id,
        ).one_or_none()
        if grant is None:
            raise TvmGrantRequiredError(tvm_client_id)
