# -*- coding: utf-8 -*-

import re

from passport.backend.vault.api.db import get_db
from passport.backend.vault.api.errors import (
    AccessError,
    NonexistentEntityError,
)
from passport.backend.vault.api.models.base import (
    BaseModel,
    ExternalRecordState,
    MagicBigInteger,
    MagicJSON,
    Timestamp,
)
import sqlalchemy
from sqlalchemy import (
    asc,
    cast,
    Index,
    or_,
    UniqueConstraint,
)
from sqlalchemy.orm import undefer


db = get_db()


class UserInfo(BaseModel):
    __tablename__ = 'user_info'
    __repr_attrs__ = ['login', 'state']

    default_serialization_columns = [
        'uid', 'first_name', 'last_name', 'login',
    ]
    default_serialization_pycolumns = ['state_name']

    uid = db.Column(MagicBigInteger, primary_key=True, autoincrement=False)
    login = db.Column(db.String(255))
    keys = db.deferred(db.Column(MagicJSON))
    first_name = db.Column(db.String(255))
    last_name = db.Column(db.String(255))

    staff_id = db.Column(MagicBigInteger, nullable=True)
    staff_info = db.relationship(
        'StaffDepartmentInfo',
        primaryjoin='UserInfo.staff_id == foreign(StaffDepartmentInfo.id)',
        lazy='joined',
        uselist=False,
        viewonly=True,
    )

    state = db.Column(db.Integer, nullable=False, default=ExternalRecordState.normal.value, server_default='0')

    def state_name(self):
        return ExternalRecordState(self.state).name if self.state != ExternalRecordState.normal.value else None

    created_at = db.Column(Timestamp(current_timestamp=True), nullable=False)

    __table_args__ = (
        UniqueConstraint('login', name='ix_user_info_login'),
        Index('idx_user_info_first_name', 'first_name'),
        Index('idx_user_info_last_name', 'last_name'),
    )

    @classmethod
    def get_by_login(cls, login, with_keys=True, raise_access_error=False):
        query = cls.query.filter(cls.login == login)
        if with_keys:
            query = query.options(undefer('keys'))
        user_info = query.one_or_none()
        if not user_info:
            if raise_access_error:
                raise AccessError()
            raise NonexistentEntityError(cls, login)
        return user_info

    @classmethod
    def get_by_id(cls, uid, with_keys=True, raise_access_error=False):
        query = cls.query.filter(cls.uid == uid)
        if with_keys:
            query = query.options(undefer('keys'))
        user_info = query.one_or_none()
        if not user_info:
            if raise_access_error:
                raise AccessError()
            raise NonexistentEntityError(cls, uid)
        return user_info

    @classmethod
    def get_by_ids(cls, ids):
        return cls.query.filter(cls.uid.in_(ids)).all()

    @classmethod
    def get_by_query(cls, query, limit=None):
        query = re.sub(r'(\s)+', r' ', query).strip()
        name_query = u'%{query}%'.format(query=query)
        login_query = u'{query}%'.format(query=query)
        return cls.query.filter(
            cls.state == ExternalRecordState.normal.value,
            or_(
                cls.last_name.ilike(name_query),
                cls.first_name.ilike(name_query),
                (cls.first_name + ' ' + cls.last_name).ilike(name_query),
                (cls.last_name + ' ' + cls.first_name).ilike(name_query),
                cls.login.ilike(login_query),
                cast(cls.uid, sqlalchemy.String) == query,
            ),
        ).order_by(
            asc(cls.login)
        ).limit(limit).all()

    def __hash__(self):
        return hash(
            (self.uid, self.login, str(list(self.keys)),
             self.first_name, self.last_name, self.staff_id, self.state),
        )

    def __eq__(self, other):
        return (
            (self.uid, self.login, str(list(self.keys)), self.first_name, self.last_name, self.staff_id, self.state) ==
            (other.uid, other.login, str(list(other.keys)), other.first_name, other.last_name, other.staff_id, other.state)
        )


class CreatorMixin(object):
    """
    from passport.backend.vault.api.models.user_info import CreatorMixin

    class CurrentModel(BaseModel, CreatorMixin):
        default_serialization_pycolumns = ['creator_login']
        creator_user_info = CreatorMixin.creator_relationship('Secret')
    """

    created_at = db.Column(Timestamp(current_timestamp=True), nullable=False)
    created_by = db.Column(MagicBigInteger, nullable=False)

    @staticmethod
    def creator_relationship(model_name, field_name='created_by', lazy='joined'):
        return db.relationship(
            'UserInfo',
            primaryjoin='{model_name}.{field_name} == foreign(UserInfo.uid)'.format(
                model_name=model_name,
                field_name=field_name,
            ),
            lazy=lazy,
            uselist=False,
            viewonly=True,
        )

    def creator_login(self):
        return self.creator_user_info.login if self.creator_user_info else None
