# -*- coding: utf-8 -*-

from enum import IntEnum
import time
import warnings

from passport.backend.vault.api.db import get_db
from passport.backend.vault.api.errors import (
    AbcRoleNotFoundError,
    AbcScopeNotFoundError,
    AbcServiceNotFoundError,
    StaffGroupNotFoundError,
    UserNotFoundError,
)
from passport.backend.vault.api.models import (
    AbcDepartmentInfo,
    AbcRole,
    AbcScope,
    StaffDepartmentInfo,
    UserInfo,
)
from passport.backend.vault.api.models.base import (
    BaseModel,
    MagicBigInteger,
    MagicInteger,
    Timestamp,
    UUIDType,
)
from passport.backend.vault.api.models.bundle import BundleUUIDType
from passport.backend.vault.api.models.external_record import (
    ExternalRecord,
    ExternalType,
)
from passport.backend.vault.api.models.secret import SecretUUIDType
from passport.backend.vault.api.utils import ulid
from sqlalchemy import (
    and_,
    exc as sa_exc,
    Index,
    join,
    select,
    union_all,
    UniqueConstraint,
)
from sqlalchemy.exc import IntegrityError


db = get_db()


class Roles(IntEnum):
    OWNER = 1
    READER = 2
    APPENDER = 3
    SUPERVISOR = 2 ** 31


class RolesWeights(IntEnum):
    # Веса для определения «эффективной роли» пользователя
    SUPERVISOR = 2 ** 31
    OWNER = 100
    READER = 20
    APPENDER = 10


class UserRole(BaseModel):
    __tablename__ = 'user_roles'
    __repr_attrs__ = ['role_id', 'external_type', 'uid', 'abc_id', 'abc_scope_id', 'staff_id', 'secret_uuid', 'bundle_uuid']

    max_serialization_depth = 1
    default_serialization_columns = [
        'abc_id', 'abc_scope_id', 'staff_id', 'uid',
        'secrets_uuid', 'created_at', 'created_by',
    ]
    default_serialization_pycolumns = [
        'role_slug', 'login', 'creator_login',
        'abc_name', 'abc_slug', 'abc_url',
        'abc_scope', 'abc_scope_name',
        'abc_role', 'abc_role_name',
        'staff_name', 'staff_slug', 'staff_url',
    ]
    remove_columns_equals_zero = ['uid', 'staff_id', 'abc_id', 'abc_scope_id', 'abc_role_id']

    id = db.Column(UUIDType, primary_key=True, default=lambda: ulid.create_ulid())

    external_type = db.Column(ExternalType, nullable=False, server_default='')

    role_id = db.Column(MagicInteger, nullable=False)

    uid = db.Column(MagicBigInteger, nullable=False, default='0')

    abc_id = db.Column(MagicBigInteger, nullable=False, default='0')
    abc_scope_id = db.Column(MagicInteger, nullable=False, default='0')
    abc_role_id = db.Column(MagicInteger, nullable=False, default='0')

    staff_id = db.Column(MagicBigInteger, nullable=False, default='0')

    secret_uuid = db.Column(SecretUUIDType(default=''))
    bundle_uuid = db.Column(BundleUUIDType(default=''))

    created_at = db.Column(Timestamp(current_timestamp=True), nullable=False)
    created_by = db.Column(MagicBigInteger, nullable=False)

    not_null_any_of_columns = ['uid', 'abc_id', 'staff_id']

    __table_args__ = (
        UniqueConstraint(
            'secret_uuid',
            'bundle_uuid',
            'role_id',
            'external_type',
            'uid',
            'staff_id',
            'abc_id',
            'abc_scope_id',
            'abc_role_id',
            name='user_roles_uq_1',
        ),
        Index('idx_user_roles_secret_uuid', 'secret_uuid', 'role_id'),
        Index('idx_user_roles_bundle_uuid', 'bundle_uuid', 'role_id'),
        Index('idx_user_roles_user', 'uid', 'secret_uuid'),
        Index('idx_user_roles_staff', 'external_type', 'staff_id', 'secret_uuid'),
        Index('idx_user_roles_abc', 'external_type', 'abc_id', 'abc_scope_id', 'abc_role_id', 'secret_uuid')
    )

    def __init__(self, *args, **kwargs):
        super(UserRole, self).__init__(*args, **kwargs)

        if self.not_null_any_of_columns:
            not_nulled = [
                v for k, v in kwargs.viewitems()
                if k in self.not_null_any_of_columns and v is not None
            ]
            if not not_nulled:
                raise ValueError('only_one: %s, presents: %s' % (
                    self.not_null_any_of_columns,
                    not_nulled,
                ))

    abc_info = db.relationship(
        'AbcDepartmentInfo',
        primaryjoin='UserRole.abc_id == foreign(AbcDepartmentInfo.id)',
        lazy='selectin',
        uselist=False,
        viewonly=True,
    )

    _abc_scope = db.relationship(
        'AbcScope',
        primaryjoin='UserRole.abc_scope_id == foreign(AbcScope.id)',
        lazy='selectin',
        uselist=False,
        viewonly=True,
    )

    _abc_role = db.relationship(
        'AbcRole',
        primaryjoin='UserRole.abc_role_id == foreign(AbcRole.id)',
        lazy='selectin',
        uselist=False,
        viewonly=True,
    )

    staff_info = db.relationship(
        'StaffDepartmentInfo',
        primaryjoin='UserRole.staff_id == foreign(StaffDepartmentInfo.id)',
        lazy='selectin',
        uselist=False,
        viewonly=True,
    )

    user_info = db.relationship(
        'UserInfo',
        primaryjoin='UserRole.uid == foreign(UserInfo.uid)',
        lazy='selectin',
        uselist=False,
        viewonly=True,
    )

    creator_user_info = db.relationship(
        'UserInfo',
        primaryjoin='UserRole.created_by == foreign(UserInfo.uid)',
        lazy='selectin',
        uselist=False,
        viewonly=True,
    )

    def login(self):
        return self.user_info.login if self.user_info else None

    def user_state(self):
        return self.user_info.state_name() if self.user_info and self.user_info.state != ExternalRecord.normal.value else None

    def creator_login(self):
        return self.creator_user_info.login if self.creator_user_info else None

    def abc_name(self):
        return self.abc_info.display_name if self.abc_info else None

    def abc_state(self):
        return self.abc_info.state_name() if self.abc_info and self.abc_info.state != ExternalRecord.normal.value else None

    def abc_scope(self):
        return self._abc_scope.unique_name if self._abc_scope else None

    def abc_scope_name(self):
        return self._abc_scope.display_name if self._abc_scope else None

    def abc_role(self):
        return self._abc_role.id if self._abc_role else None

    def abc_role_name(self):
        return self._abc_role.display_name if self._abc_role else None

    def abc_slug(self):
        return self.abc_info.unique_name if self.abc_info else None

    def abc_url(self):
        slug = self.abc_slug()
        return self.config['abc']['url_template'].format(slug=slug) if slug else None

    def abc_scope_slug(self):
        return self.abc_scope.unique_name if self.abc_info else None

    def staff_name(self):
        return self.staff_info.display_name if self.staff_info else None

    def staff_slug(self):
        return self.staff_info.unique_name if self.staff_info else None

    def staff_url(self):
        slug = self.staff_slug()
        return self.config['staff']['url_template'].format(slug=slug) if slug else None

    def staff_state(self):
        return self.staff_info.state_name() if self.staff_info and self.staff_info.state != ExternalRecord.normal.value else None

    @property
    def role(self):
        return Roles(self.role_id)

    def role_slug(self):
        return Roles(self.role_id).name

    @staticmethod
    def get_abc_scope(abc_id, abc_scope, raises=True):
        if abc_id is None or abc_scope is None:
            if raises:
                if abc_id is None:
                    raise AbcServiceNotFoundError(abc_id)
                raise AbcScopeNotFoundError(abc_id, abc_scope)
            return None

        service = AbcDepartmentInfo.query.filter(
            AbcDepartmentInfo.id == abc_id,
        ).one_or_none()
        if not service:
            if raises:
                raise AbcServiceNotFoundError(abc_id)
            return

        scope = filter(
            lambda x: x.unique_name == abc_scope,
            service.scopes,
        )

        if not scope:
            if raises:
                raise AbcScopeNotFoundError(abc_id, abc_scope)
            return None

        return scope[0]

    @staticmethod
    def get_abc_role(abc_id, abc_role_id, raises=True):
        if abc_id is None or abc_role_id is None:
            if raises:
                if abc_id is None:
                    raise AbcServiceNotFoundError(abc_id)
                raise AbcRoleNotFoundError(abc_id, abc_role_id)
            return None

        service = AbcDepartmentInfo.query.filter(
            AbcDepartmentInfo.id == abc_id,
        ).one_or_none()
        if not service:
            if raises:
                raise AbcServiceNotFoundError(abc_id)
            return

        role = filter(
            lambda x: x.id == int(abc_role_id),
            service.roles,
        )

        if not role:
            if raises:
                raise AbcRoleNotFoundError(abc_id, abc_role_id)
            return None

        return role[0]

    @staticmethod
    def create_user_role(
        creator_uid,
        secret,
        role,
        abc_id=None,
        abc_scope=None,
        abc_role_id=None,
        staff_id=None,
        uid=None,
        validate_uid=True,
    ):
        current_time = time.time()
        roles_kwargs = dict(
            secret_uuid=secret.uuid,
            role_id=role.value,
            created_by=creator_uid,
            created_at=current_time,
        )

        if abc_id is not None:
            if abc_role_id is not None:
                role = UserRole.get_abc_role(abc_id, abc_role_id)
                user_role = UserRole(
                    external_type='abc',
                    abc_id=abc_id if role is not None else None,
                    abc_role_id=role.id if role is not None else None,
                    **roles_kwargs
                )
            else:
                scope = UserRole.get_abc_scope(abc_id, abc_scope)
                user_role = UserRole(
                    external_type='abc',
                    abc_id=abc_id if scope is not None else None,
                    abc_scope_id=scope.id if scope is not None else None,
                    **roles_kwargs
                )
        elif staff_id is not None:
            staff_group = StaffDepartmentInfo.query.filter(
                StaffDepartmentInfo.id == staff_id,
            ).one_or_none()
            if staff_group is None:
                raise StaffGroupNotFoundError(staff_id)
            user_role = UserRole(
                external_type='staff',
                staff_id=staff_group.id,
                **roles_kwargs
            )
        elif uid is not None:
            if validate_uid:
                user = UserInfo.query.filter(
                    UserInfo.uid == uid,
                ).one_or_none()
                if user is None:
                    raise UserNotFoundError(uid)
            user_role = UserRole(
                external_type='user',
                uid=uid,
                **roles_kwargs
            )
        else:
            raise Exception('Invalid roles options')

        secret.touch(updated_at=current_time, updated_by=creator_uid)
        return user_role

    @staticmethod
    def delete_user_role(deleter_uid, secret, role, abc_id=None, abc_scope=None, abc_role_id=None, staff_id=None, uid=None):
        filters = and_(
            UserRole.secret_uuid == secret.uuid,
            UserRole.role_id == role.value,
        )
        if abc_id is not None:
            if abc_role_id is not None:
                role = AbcRole.get_by_id(abc_role_id)
                filters = filters & and_(
                    UserRole.abc_id == abc_id,
                    UserRole.abc_role_id == abc_role_id,
                    UserRole.abc_scope_id == 0,
                )
            else:
                scope = AbcScope.get_by_name(abc_scope)
                filters = filters & and_(
                    UserRole.abc_id == abc_id,
                    UserRole.abc_scope_id == scope.id,
                    UserRole.abc_role_id == 0,
                )

        if staff_id is not None:
            filters = filters & (
                UserRole.staff_id == staff_id
            )

        if uid is not None:
            filters = filters & (
                UserRole.uid == uid
            )

        user_role = UserRole.query.filter(filters).one_or_none()
        if user_role:
            get_db().session.delete(user_role)
            secret.touch(updated_by=deleter_uid)
        return user_role

    @staticmethod
    def get_identity_role_filters(uid, roles=None, bundle_uuid=None, secret_uuid=None):
        filters = and_()
        if roles:
            if not isinstance(roles, (list, tuple)):
                roles = [roles]
            if Roles['READER'] in roles:
                roles.append(Roles['OWNER'])
            filters = filters & (UserRole.role_id.in_([role.value for role in roles]))
        if bundle_uuid:
            filters = filters & (UserRole.bundle_uuid == bundle_uuid)
        if secret_uuid:
            filters = filters & (UserRole.secret_uuid == secret_uuid)

        roles_fields = [
            UserRole.id.label('user_roles_id'),
            UserRole.secret_uuid.label('user_roles_secret_uuid'),
            UserRole.bundle_uuid.label('user_roles_bundle_uuid'),
        ]
        if uid is not None:
            query = union_all(
                select(
                    roles_fields,
                ).select_from(
                    UserRole,
                ).where(
                    and_(
                        UserRole.external_type == 'user',
                        UserRole.uid == uid,
                        filters,
                    ),
                ),
                select(
                    roles_fields,
                ).select_from(
                    join(
                        UserRole,
                        ExternalRecord,
                        and_(
                            UserRole.external_type == 'staff',
                            UserRole.external_type == ExternalRecord.external_type,
                            UserRole.staff_id == ExternalRecord.external_id,
                            ExternalRecord.uid == uid,
                            UserRole.abc_scope_id == ExternalRecord.external_scope_id,
                            UserRole.abc_role_id == ExternalRecord.external_role_id,
                        ),
                    ),
                ).where(
                    filters,
                ),
                select(
                    roles_fields,
                ).select_from(
                    join(
                        UserRole,
                        ExternalRecord,
                        and_(
                            UserRole.external_type == 'abc',
                            UserRole.external_type == ExternalRecord.external_type,
                            UserRole.abc_id == ExternalRecord.external_id,
                            UserRole.abc_scope_id == ExternalRecord.external_scope_id,
                            UserRole.abc_role_id == ExternalRecord.external_role_id,
                            ExternalRecord.uid == uid,
                        ),
                    ),
                ).where(
                    filters,
                ),
            ).alias('user_roles_union')
        else:
            query = select(
                roles_fields,
            ).select_from(
                UserRole,
            ).where(
                filters,
            ).alias('user_roles_union')

        return query

    @staticmethod
    def _get_roles_query(uid=None, roles=None, bundle_uuid=None, secret_uuid=None):
        filtered_roles = UserRole.get_identity_role_filters(
            uid,
            roles=roles,
            secret_uuid=secret_uuid,
            bundle_uuid=bundle_uuid,
        )
        query = UserRole.query.join(
            filtered_roles,
            UserRole.id == filtered_roles.c.user_roles_id,
        )
        return query

    @staticmethod
    def get_roles(uid=None, roles=None, bundle_uuid=None, secret_uuid=None):
        roles = UserRole._get_roles_query(
            uid=uid, roles=roles, bundle_uuid=bundle_uuid, secret_uuid=secret_uuid,
        ).all()
        return list(roles)

    @staticmethod
    def get_roles_count(uid=None, roles=None, bundle_uuid=None, secret_uuid=None):
        return UserRole._get_roles_query(
            uid=uid, roles=roles, bundle_uuid=bundle_uuid, secret_uuid=secret_uuid,
        ).count()

    @staticmethod
    def safely_add_user_role_to_secret(user_role):
        with warnings.catch_warnings():
            warnings.simplefilter("ignore", category=sa_exc.SAWarning)
            try:
                db = get_db()
                with db.session.begin_nested():
                    db.session.add(user_role)
                db.session.commit()
                return True, user_role
            except IntegrityError:
                return False, user_role
