# coding: utf-8

from collections import OrderedDict
from datetime import datetime
import time

from passport.backend.vault.api.db import get_db
from passport.backend.vault.api.models import (
    DelegationToken,
    Secret,
    SecretVersion,
)
from passport.backend.vault.api.models.base import State
from sqlalchemy import (
    between,
    distinct,
    func,
)
import yenv


def unpack_metrics(metrics):
    return [dict(name=k, val=v) for k, v in metrics.items()]


def make_golovan_metrics(config, metrics, ttl=None):
    if ttl is None:
        ttl = config['yasm_agent']['default_ttl']

    return [{
        'tags': {
            'itype': config['yasm_agent']['itype'],
            'ctype': yenv.type,
            'prj': config['yasm_agent']['prj'],
            'geo': config['application']['current_dc'] or 'none',
        },
        'ttl': ttl,
        'values': unpack_metrics(metrics),
    }]


def make_xunistater_metrics(config, metrics):
    return dict(
        (k, dict(value=v)) for k, v in metrics.items()
    )


def get_day_utc_period(ts=None):
    if not ts:
        ts = time.time()
    start = time.mktime(datetime.utcfromtimestamp(ts).replace(hour=0, minute=0, second=0, microsecond=0).timetuple())
    end = start + 24 * 60 * 60 - 0.001
    return (start, end)


def get_today_counts():
    period = get_day_utc_period()
    metrics = [
        (Secret, 'today_count_secrets', 'today_secrets_creators',),
        (SecretVersion, 'today_count_versions', 'today_versions_creators',),
        (DelegationToken, 'today_count_tokens', 'today_tokens_creators',),
    ]

    result = []
    for m in metrics:
        counts = get_db().session.query(
            func.count(),
            func.count(distinct(m[0].created_by)),
        ).select_from(m[0]).filter(
            between(m[0].created_at, period[0], period[1]),
        ).first()
        result.extend([
            (m[1], counts[0]),
            (m[2], counts[1]),
        ])

    return result


def get_global_metrics():
    result = [
        ('count_secrets', Secret.query.count()),
        ('count_active_secrets', Secret.query.filter(Secret.state == State.normal.value).count()),
        ('count_hidden_secrets', Secret.query.filter(Secret.state == State.hidden.value).count()),

        ('count_versions', SecretVersion.query.count()),
        ('count_active_versions', SecretVersion.query.filter(SecretVersion.state == State.normal.value).count()),
        ('count_hidden_versions', SecretVersion.query.filter(SecretVersion.state == State.hidden.value).count()),

        ('count_tokens', DelegationToken.get_count('normal')),
        ('count_revoked_tokens', DelegationToken.get_count('revoked'))
    ]
    result.extend(get_today_counts())
    return OrderedDict(result)
