# -*- coding: utf-8 -*-
from passport.backend.utils.common import (
    merge_dicts,
    noneless_dict,
)
from passport.backend.utils.tskv import TskvLogEntry
from sqlalchemy.exc import DontWrapMixin


class BaseError(Exception):
    """
    Базовая ошибка
    """
    status_code = 500
    code = 'base_error'
    info = {}
    private_info = {}
    msg = None
    level = 'WARNING'

    def __init__(self, msg=None, private_info=None, **kwargs):
        """
        :param private_info: поля, которые будут записаны в логи, но не будут переданы пользователю в ответ на запрос
        :param kwargs: поля, которые будут записаны в логи и переданы в ответ на запрос
        """
        self.msg = msg or self.msg
        self.private_info = private_info or {}
        self.kwargs = kwargs
        if msg:
            self.private_info.update({'msg': msg})

    def as_dict(self):
        return noneless_dict(dict(
            status='error',
            code=self.code,
            message=self.msg,
            **merge_dicts(self.info, self.kwargs)
        ))

    def as_private_dict(self):
        return merge_dicts(self.as_dict(), self.private_info)

    def as_tskv(self):
        return TskvLogEntry(**self.as_dict())

    def __str__(self):
        return str(self.as_tskv())


class BaseUUIDError(BaseError):
    pass


class InvalidUUIDPrefix(BaseUUIDError, DontWrapMixin):
    """
    Передан неверный префикс для UUID
    """
    status_code = 400
    code = 'invalid_uuid_prefix'
    msg = 'Invalid id prefix'


class InvalidUUIDValue(BaseUUIDError, DontWrapMixin):
    """
    Передано неверное значение UUID
    """
    status_code = 400
    code = 'invalid_uuid_value'
    msg = 'Invalid id value'
