# -*- coding: utf-8 -*-

from base64 import (
    urlsafe_b64decode,
    urlsafe_b64encode,
)
import binascii
import hashlib
import os

import cityhash
from paramiko import (
    DSSKey,
    ECDSAKey,
    Message,
    RSAKey,
)


_key_types = {
    'ssh-rsa': RSAKey,
    'ssh-dss': DSSKey,
    'ecdsa-sha2-nistp256': ECDSAKey,
    'ecdsa-sha2-nistp384': ECDSAKey,
    'ecdsa-sha2-nistp521': ECDSAKey,
}


def default_hash(value):
    return hashlib.sha256(value).hexdigest()


def _message_to_bytes(msg):
    if isinstance(msg, Message):
        return msg.asbytes()
    else:
        return bytes(msg)


def sign_in_memory(data, key):
    return urlsafe_b64encode(_message_to_bytes(key.sign_ssh_data(data)))


def detect_key_type(tag):
    return _key_types.get(tag)


def verify_signature_v1(data, signature, public_keys):
    hash = default_hash(data)
    for public_key in public_keys:
        splitted_key = public_key.split(' ')
        tag, public_key_content = splitted_key[0], splitted_key[1]
        key_type = detect_key_type(tag)
        if not key_type:
            continue
        pk = key_type(data=urlsafe_b64decode(str(public_key_content)))
        if pk.verify_ssh_sig(hash, Message(urlsafe_b64decode(signature))):
            return True
    return False


def verify_signature_v2(data, signature, public_keys):
    for public_key in public_keys:
        splitted_key = public_key.split(' ')
        tag, public_key_content = splitted_key[0], splitted_key[1]
        key_type = detect_key_type(tag)
        if not key_type:
            continue
        pk = key_type(data=urlsafe_b64decode(str(public_key_content)))
        if pk.verify_ssh_sig(data.encode('utf-8'), Message(urlsafe_b64decode(signature))):
            return True
    return False


def verify_signature_v3(data, signature, public_keys):
    hash = default_hash(data)
    for public_key in public_keys:
        splitted_key = public_key.split(' ')
        tag, public_key_content = splitted_key[0], splitted_key[1]
        key_type = detect_key_type(tag)
        if not key_type:
            continue
        pk = key_type(data=urlsafe_b64decode(str(public_key_content)))
        if pk.verify_ssh_sig(hash, Message(urlsafe_b64decode(signature))):
            return True
    return False


def token_bytes(nbytes=None):  # pragma: no cover
    """Return a random byte string containing *nbytes* bytes.
    If *nbytes* is ``None`` or not supplied, a reasonable
    default is used.
    >>> token_bytes(16)  #doctest:+SKIP
    b'\\xebr\\x17D*t\\xae\\xd4\\xe3S\\xb6\\xe2\\xebP1\\x8b'
    """
    if nbytes is None:
        nbytes = 32
    return os.urandom(nbytes)


def token_urlsafe(nbytes=None, strip=True):  # pragma: no cover
    """Return a random URL-safe text string, in Base64 encoding.
    The string has *nbytes* random bytes.  If *nbytes* is ``None``
    or not supplied, a reasonable default is used.
    >>> token_urlsafe(16)  #doctest:+SKIP
    'Drmhze6EPcv0fN_81Bj-nA'
    """
    tok = token_bytes(nbytes)
    tok = urlsafe_b64encode(tok)
    if strip:
        return tok.rstrip(b'=')
    return tok


def token_hex(nbytes=None):
    """Return a random text string, in hexadecimal.

    The string has *nbytes* random bytes, each byte converted to two
    hex digits.  If *nbytes* is ``None`` or not supplied, a reasonable
    default is used.

    >>> token_hex(16)  #doctest:+SKIP
    'f9bf78b9a18ce6d46a0cd2b0b86df9da'

    """
    return binascii.hexlify(token_bytes(nbytes)).decode('ascii')


class InvalidSignedString(Exception):
    pass


def hash_string(str_):
    return '{:0x}'.format(cityhash.hash64(str_.encode('utf-8')))


def sign_string(payload):
    return u'{payload}.{version}.{sign}'.format(
        payload=payload,
        version='1',
        sign=hash_string(payload),
    )


def parse_signed_string(signed_string, verify=True):
    parts = signed_string.rsplit('.', 2)
    if verify and (
        len(parts) != 3
        or parts[1] != '1'
        or parts[2].lower() != hash_string(parts[0])
    ):
        raise InvalidSignedString()
    return parts[0]
