# -*- coding: utf-8 -*-

"""
Работа с уникальными лексикографически-сортируемыми идентификаторами
https://st.yandex-team.ru/PASSP-19614

 01an4z07by      79ka1307sr9x4mv3

|----------|    |----------------|
 Timestamp          Randomness
   48bits             80bits

https://github.com/ulid/spec
"""
from __future__ import absolute_import

from functools import total_ordering
import os
import sys
import time

from passport.backend.vault.api.utils.errors import (
    InvalidUUIDPrefix,
    InvalidUUIDValue,
)
import six


MILLISECS_IN_SECS = 1000


def create_ulid(value=None, prefix=None):
    return ULID(value, prefix=prefix)


def current_timestamp():
    return int(time.time() * MILLISECS_IN_SECS)


def get_randomness(len):
    return os.urandom(len)


def to_byte_string(l):
    if six.PY3:
        return bytes(l)

    return bytes(b''.join(chr(b) for b in l))


def int_to_bytes(value, length, byteorder=None):
    if not isinstance(value, six.integer_types):
        raise TypeError()
    byteorder = byteorder or sys.byteorder
    sequence = [value >> (i * 8) & 0xFF for i in range(length)]
    if byteorder == 'big':
        sequence = reversed(sequence)
    return to_byte_string(sequence)


def bytes_to_int(value, byteorder=None):
    if not isinstance(value, bytes):
        raise TypeError()
    byteorder = byteorder or sys.byteorder
    value = bytearray(value)

    if six.PY3:
        return int.from_bytes(value, byteorder)

    if byteorder == 'little':
        value.reverse()
    return int(bytes(value).encode('hex'), 16)


def string_to_bytes(string):
    if six.PY3:
        return six.ensure_binary(string)
    return [ord(b) for b in string]


class GenerateULIDBytes(object):
    TIMESTAMP_LEN = 6
    RANDOMNESS_LEN = 10
    BYTES_ORDER = 'big'

    def __init__(self):
        self._last_time = None
        self._randomness = None

    def __call__(self):
        timestamp = current_timestamp()

        if self._last_time is not None and timestamp == self._last_time:
            randomness = int_to_bytes(bytes_to_int(self._randomness, self.BYTES_ORDER) + 1, self.RANDOMNESS_LEN, self.BYTES_ORDER)
        else:
            self._last_time = timestamp
            randomness = get_randomness(self.RANDOMNESS_LEN)
        self._randomness = randomness

        return int_to_bytes(timestamp, self.TIMESTAMP_LEN, self.BYTES_ORDER) + randomness


class CrockfordBase32ULIDCodec(object):
    ENCODE = '0123456789abcdefghjkmnpqrstvwxyz'
    DECODE = [
        0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
        0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
        0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
        0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
        0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x00, 0x01,
        0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0xFF, 0xFF,
        0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E,
        0x0F, 0x10, 0x11, 0xFF, 0x12, 0x13, 0xFF, 0x14, 0x15, 0xFF,
        0x16, 0x17, 0x18, 0x19, 0x1A, 0xFF, 0x1B, 0x1C, 0x1D, 0x1E,
        0x1F, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x0A, 0x0B, 0x0C,
        0x0D, 0x0E, 0x0F, 0x10, 0x11, 0xFF, 0x12, 0x13, 0xFF, 0x14,
        0x15, 0xFF, 0x16, 0x17, 0x18, 0x19, 0x1A, 0xFF, 0x1B, 0x1C,
        0x1D, 0x1E, 0x1F, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    ]
    BYTES_LEN = 16
    TIMESTAMP_LEN = 6
    RANDOMNESS_LEN = 10

    TIMESTAMP_REPR_LEN = 10
    RANDOMNESS_REPR_LEN = 16
    REPR_LEN = TIMESTAMP_REPR_LEN + RANDOMNESS_REPR_LEN

    def encode(self, binary):
        if len(binary) != self.BYTES_LEN:
            raise ValueError('ULID has to be exactly 16 bytes long ({})'.format(len(binary)))
        return (self._encode_timestamp(binary[:self.TIMESTAMP_LEN]) +
                self._encode_randomness(binary[self.TIMESTAMP_LEN:]))

    def decode(self, encoded):
        if len(encoded) != self.REPR_LEN:
            raise ValueError('Encoded ULID has to be exactly 26 characters long.')
        return (self._decode_timestamp(encoded[:self.TIMESTAMP_REPR_LEN]) +
                self._decode_randomness(encoded[self.TIMESTAMP_REPR_LEN:]))

    def _encode_timestamp(self, binary):
        if len(binary) != self.TIMESTAMP_LEN:
            raise ValueError('Timestamp value has to be exactly 6 bytes long.')
        lut = self.ENCODE
        values = string_to_bytes(binary)
        return ''.join([
            lut[(values[0] & 224) >> 5],
            lut[(values[0] & 31)],
            lut[(values[1] & 248) >> 3],
            lut[((values[1] & 7) << 2) | ((values[2] & 192) >> 6)],
            lut[((values[2] & 62) >> 1)],
            lut[((values[2] & 1) << 4) | ((values[3] & 240) >> 4)],
            lut[((values[3] & 15) << 1) | ((values[4] & 128) >> 7)],
            lut[(values[4] & 124) >> 2],
            lut[((values[4] & 3) << 3) | ((values[5] & 224) >> 5)],
            lut[(values[5] & 31)],
        ])

    def _encode_randomness(self, binary):
        if len(binary) != self.RANDOMNESS_LEN:
            raise ValueError('Randomness value has to be exactly 10 bytes long.')
        lut = self.ENCODE
        values = string_to_bytes(binary)
        return ''.join([
            lut[(values[0] & 248) >> 3],
            lut[((values[0] & 7) << 2) | ((values[1] & 192) >> 6)],
            lut[(values[1] & 62) >> 1],
            lut[((values[1] & 1) << 4) | ((values[2] & 240) >> 4)],
            lut[((values[2] & 15) << 1) | ((values[3] & 128) >> 7)],
            lut[(values[3] & 124) >> 2],
            lut[((values[3] & 3) << 3) | ((values[4] & 224) >> 5)],
            lut[(values[4] & 31)],
            lut[(values[5] & 248) >> 3],
            lut[((values[5] & 7) << 2) | ((values[6] & 192) >> 6)],
            lut[(values[6] & 62) >> 1],
            lut[((values[6] & 1) << 4) | ((values[7] & 240) >> 4)],
            lut[((values[7] & 15) << 1) | ((values[8] & 128) >> 7)],
            lut[(values[8] & 124) >> 2],
            lut[((values[8] & 3) << 3) | ((values[9] & 224) >> 5)],
            lut[(values[9] & 31)],
        ])

    def _decode_timestamp(self, encoded):
        if len(encoded) != self.TIMESTAMP_REPR_LEN:
            raise ValueError('ULID timestamp has to be exactly 10 characters long.')
        lut = self.DECODE
        values = string_to_bytes(encoded)
        return to_byte_string([
            ((lut[values[0]] << 5) | lut[values[1]]) & 0xFF,
            ((lut[values[2]] << 3) | (lut[values[3]] >> 2)) & 0xFF,
            ((lut[values[3]] << 6) | (lut[values[4]] << 1) | (lut[values[5]] >> 4)) & 0xFF,
            ((lut[values[5]] << 4) | (lut[values[6]] >> 1)) & 0xFF,
            ((lut[values[6]] << 7) | (lut[values[7]] << 2) | (lut[values[8]] >> 3)) & 0xFF,
            ((lut[values[8]] << 5) | (lut[values[9]])) & 0xFF,
        ])

    def _decode_randomness(self, encoded):
        if len(encoded) != self.RANDOMNESS_REPR_LEN:
            raise ValueError('ULID randomness has to be exactly 16 characters long.')
        lut = self.DECODE
        values = string_to_bytes(encoded)
        try:
            return to_byte_string([
                ((lut[values[0]] << 3) | (lut[values[1]] >> 2)) & 0xFF,
                ((lut[values[1]] << 6) | (lut[values[2]] << 1) | (lut[values[3]] >> 4)) & 0xFF,
                ((lut[values[3]] << 4) | (lut[values[4]] >> 1)) & 0xFF,
                ((lut[values[4]] << 7) | (lut[values[5]] << 2) | (lut[values[6]] >> 3)) & 0xFF,
                ((lut[values[6]] << 5) | (lut[values[7]])) & 0xFF,
                ((lut[values[8]] << 3) | (lut[values[9]] >> 2)) & 0xFF,
                ((lut[values[9]] << 6) | (lut[values[10]] << 1) | (lut[values[11]] >> 4)) & 0xFF,
                ((lut[values[11]] << 4) | (lut[values[12]] >> 1)) & 0xFF,
                ((lut[values[12]] << 7) | (lut[values[13]] << 2) | (lut[values[14]] >> 3)) & 0xFF,
                ((lut[values[14]] << 5) | (lut[values[15]])) & 0xFF,
            ])
        except IndexError:
            raise ValueError('ULID contains bytes out of proper range')


@total_ordering
class ULID(object):
    DEFAULT_ULID_BYTES_GENERATOR = GenerateULIDBytes()
    DEFAULT_ULID_CODEC = CrockfordBase32ULIDCodec()
    BYTES_LEN = 16
    CHAR_LEN = 26

    def __init__(self, value=None, bytes_=None, generator=None, codec=None, prefix=None, ignore_prefix=False):
        self._generator = generator or self.DEFAULT_ULID_BYTES_GENERATOR
        self._codec = codec or self.DEFAULT_ULID_CODEC
        self.prefix = prefix.lower() if prefix else None
        self.ignore_prefix = ignore_prefix

        try:
            self.bytes = None
            if bytes_ is not None:
                self.bytes = six.ensure_binary(bytes_)
                if len(self.bytes) != self.BYTES_LEN:
                    raise ValueError()
            elif not value and not isinstance(value, six.string_types):
                self.bytes = self._generator()
            elif isinstance(value, ULID):
                self.bytes = six.ensure_binary(value.bytes)
                self.prefix = prefix or value.prefix
            elif isinstance(value, six.string_types):
                if len(value) == 36:
                    try:
                        self.bytes = self._parse_uuid_string(value)
                    except ValueError:
                        pass

                if not self.bytes:
                    value = value.lower()
                    value = self._strip_prefix(value)
                    self.bytes = self._codec.decode(value)
            else:
                raise ValueError()
        except InvalidUUIDPrefix:
            raise
        except ValueError:
            raise InvalidUUIDValue('{:s} is an invalid UUID value'.format(repr(value)))

    def _strip_prefix(self, value_str):
        if self.ignore_prefix and len(value_str) > self.CHAR_LEN:
            return value_str[len(value_str) - self.CHAR_LEN:]
        elif self.prefix and value_str.find('-') >= 0:
            prefix = value_str[:len(self.prefix)]
            if not self.ignore_prefix and prefix != self.prefix:
                raise InvalidUUIDPrefix('{} is an invalid UUID prefix ({}, {})'.format(prefix, value_str, self.prefix))
            return value_str[len(prefix) + 1:]
        else:
            return value_str

    def _parse_uuid_string(self, str):
        bytes_ = bytes(bytearray.fromhex(str.replace('-', '')))
        return bytes_

    def str(self, skip_prefix=False):
        if self.prefix and not skip_prefix:
            return self.prefix + '-' + self._codec.encode(self.bytes)
        else:
            return self._codec.encode(self.bytes)

    def __str__(self):
        return self.str()

    def __repr__(self):
        class_name = self.__class__.__name__
        tail = ', ignore_prefix=True' if self.ignore_prefix else ''
        if self.prefix:
            tail += ', prefix=\'{}\''.format(self.prefix,)
        return '{}(\'{}\'{})'.format(class_name, self.__str__(), tail)

    def __int__(self):
        return bytes_to_int(self.bytes, 'big')

    def __hash__(self):
        return self.__int__()

    def __eq__(self, other):
        if isinstance(other, self.__class__):
            return self.bytes == other.bytes
        else:
            if self.str() == str(other):
                return True
            return self.str(skip_prefix=True) == str(other)

    def __lt__(self, other):
        if isinstance(other, self.__class__):
            return self.bytes < other.bytes
        else:
            return self.str() < str(other)

    @property
    def uuid(self):
        if six.PY3:
            s = self.bytes.hex()
        else:
            s = self.bytes.encode('hex')
        s = s.lower()
        return '-'.join((s[0:8], s[8:12], s[12:16], s[16:20], s[20:]))
