# -*- coding: utf-8 -*-

import itertools

from flask_wtf import FlaskForm
from passport.backend.utils import common
from passport.backend.vault.api.utils import ulid
from passport.backend.vault.api.utils.errors import BaseUUIDError
from passport.backend.vault.api.views import validators as vault_validators
import six
from wtforms import (
    Field,
    FieldList as FlaskFieldList,
    FormField,
    IntegerField as OldIntegerField,
    validators,
    widgets,
)
import wtforms_json


wtforms_json.init()


class BaseForm(FlaskForm):
    def statbox_data(self, prefix='form_'):
        """
        Возвращает данные формы для записи в statbox.log
        """
        return common.noneless_dict(dict(
            [(prefix + k, v) for k, v in self.data.iteritems()]
        ))


class StringField_(Field):
    """
    Копия StringField из WTForms. Нужно, если мы хотим, чтобы wtforms-json
    не превращал пустую строку в None.
    """
    widget = widgets.TextInput()

    def __init__(self, label='', validators=None, strip=False, **kwargs):
        super(StringField_, self).__init__(label, validators, **kwargs)
        self._strip = strip

    def process_formdata(self, valuelist):
        super(StringField_, self).process_formdata(valuelist)

        if self._strip and isinstance(self.data, six.text_type):
            self.data = self.data.strip()

    def _value(self):
        return six.text_type(self.data) if self.data is not None else ''


class IntegerField(OldIntegerField):
    def process_formdata(self, valuelist):
        if valuelist:
            form_value = valuelist[0]
            try:
                self.data = int(form_value)
            except ValueError:
                self.data = form_value
                raise ValueError('not_integer')


class ULIDField(StringField_):
    ulid_prefix = None

    def process_formdata(self, valuelist):
        if valuelist:
            form_value = valuelist[0]
            try:
                self.data = ulid.ULID(
                    form_value,
                    prefix=self.ulid_prefix,
                )
            except BaseUUIDError as e:
                self.data = form_value
                raise ValueError(e.code)


class SecretUUIDField(ULIDField):
    ulid_prefix = 'sec'


class VersionUUIDField(ULIDField):
    ulid_prefix = 'ver'


class OptionalFormField(FormField):
    def process(self, formdata, *args, **kwargs):
        self._formdata = formdata
        return super(OptionalFormField, self).process(formdata, *args, **kwargs)

    def validate(self, *args, **kwargs):
        if self._formdata:
            for field_name in self._formdata.keys():
                if field_name.startswith(self.name + self.separator):
                    return super(OptionalFormField, self).validate(*args, **kwargs)
        return True


class FieldList(FlaskFieldList):
    def validate(self, form, extra_validators=tuple()):
        """
        Переопределяем validate, чтобы он возвращал еще и индекс поля, в котором случилась ошибка.
        """
        self.errors = []

        # Run validators on all entries within
        for index, subfield in enumerate(self.entries):
            if not subfield.validate(form):
                errors = {'index': index}
                errors.update(subfield.errors)
                self.errors.append(errors)

        chain = itertools.chain(self.validators, extra_validators)
        self._run_validation_chain(form, chain)

        return len(self.errors) == 0


class PaginationFormMixin(object):
    page = IntegerField(
        'Page',
        description='page number to return, counting starts from 0',
        default=0,
        validators=[validators.Optional(), vault_validators.GreatOrEqualZeroInt()],
    )
    page_size = IntegerField(
        'Page size',
        description='page size to return',
        default=50,
        validators=[validators.Optional(), vault_validators.GreatOrEqualZeroInt()],
    )


class CommaSeparatedField(Field):
    DEFAULT_TAGS_SEPARATOR = ','
    widget = widgets.TextInput()

    def __init__(self, label='', validators=None, separator=DEFAULT_TAGS_SEPARATOR, **kwargs):
        super(CommaSeparatedField, self).__init__(label, validators, **kwargs)
        self.separator = separator

    def _value(self):  # pragma: no cover
        if self.data:
            return (self.separator + ' ').join(self.data)
        else:
            return ''

    def process_formdata(self, valuelist):
        if valuelist:
            if isinstance(valuelist[0], (list, tuple)):
                self.data = valuelist[0]
            else:
                self.data = [x.strip() for x in valuelist[0].split(self.separator)]
        else:
            self.data = None
