# -*- coding: utf-8 -*-

from passport.backend.vault.api.db import get_db
from passport.backend.vault.api.errors import AccessError
from passport.backend.vault.api.models import (
    Bundle,
    bundle_version_to_secret_version,
    BundleVersion,
    ExternalRecord,
    Roles,
    SecretVersion,
    UserRole,
)
from passport.backend.vault.api.models.base import State
from passport.backend.vault.api.views.base_view import BaseView
from passport.backend.vault.api.views.bundles.forms import (
    CreateBundleForm,
    CreateBundleVersionForm,
    GetBundleForm,
    ListBundlesForm,
    UpdateBundleForm,
)
from sqlalchemy import and_


class BaseBundleView(BaseView):
    required_user_auth = True

    # Отключили бандловые ручки в доке, пока ручки не потребуются потребителям
    # Не забыть включить тесты доки бандлов в test_docs.py
    autodoc = False


class GetBundleView(BaseBundleView):
    """
    Вернуть секрет и список его версий
    -----
    returns = [
        'bundle',
        'page',
        'page_size',
    ]
    riases = [NonexistentEntityError]
    example = {
        'arguments': {
            '<string:bundle_uuid>': 'bun-0000000000000000000003t280',
        },
        'response': {
            'bundle': {
                'bundle_roles': [{
                    'created_at': 1445385618.0,
                    'created_by': 100,
                    'creator_login': 'vault-test-100',
                    'login': 'vault-test-100',
                    'role_slug': 'OWNER',
                    'uid': 100,
                }],
                'bundle_versions': [{
                    'created_at': 1445385618.0,
                    'created_by': 100,
                    'creator_login': 'vault-test-100',
                    'secret_versions': [
                        {
                            'created_at': 1445385602.0,
                            'created_by': 100,
                            'creator_login': 'vault-test-100',
                            'keys': ['password'],
                            'secret_name': 'secret_1',
                            'value': [{'key': 'password', 'value': '123'}],
                            'version': 'ver-0000000000000000000000ygj3',
                        },
                        {
                            'created_at': 1445385607.0,
                            'created_by': 100,
                            'creator_login': 'vault-test-100',
                            'keys': ['password'],
                            'secret_name': 'secret_2',
                            'value': [{'key': 'password', 'value': '123'}],
                            'version': 'ver-0000000000000000000000ygj9',
                        },
                    ],
                    'version': 'bve-0000000000000000000003t282'
                }],
                'comment': 'All secrets',
                'created_at': 1445385618.0,
                'created_by': 100,
                'creator_login': 'vault-test-100',
                'name': 'passport_top_bundle_1',
                'updated_at': 1445385618.0,
                'updated_by': 100,
                'uuid': 'bun-0000000000000000000003t280',
            },
            'page': 0,
            'page_size': 50,
            'status': 'ok',
        }
    }
    """
    form = GetBundleForm
    statbox_mode = 'bundle'
    use_slave = True

    def validate_access(self, bundle):
        is_supervisor = self.check_if_supervisor(raises=False)
        if not is_supervisor:
            try:
                self.check_if_has_role(role=Roles.READER, bundle_uuid=bundle.uuid)
            except AccessError:
                raise AccessError(bundle_state_name=bundle.state_name())

    def process_request(self, bundle_uuid):
        bundle, bundle_versions = Bundle.get_bundle_with_versions(
            bundle_uuid,
            self.processed_form.page.data,
            self.processed_form.page_size.data,
        )

        self.validate_access(bundle)

        self.check_state(bundle, self.response_values)

        included_fields = []
        if bundle.state != State.normal.value:
            included_fields.append('state_name')

        bundle = bundle.serialize(
            max_depth=3,
            include=included_fields,
            exclude=['bundle_versions'],
        )
        bundle['bundle_versions'] = bundle_versions

        self.response_values.update({
            'page': self.processed_form.page.data,
            'page_size': self.processed_form.page_size.data,
            'bundle': bundle,
        })


class CreateBundleView(BaseBundleView):
    """
    Создать новый бандл
    -----
    returns = ['uuid']
    raises = []
    example = {
        'data': 'name=passport_top_bundle_1&comment=Dont%2Ftouch',
        'response': {
            'uuid': 'bun-0000000000000000000000ygj0',
            'status': 'ok'
        }
    }
    """
    form = CreateBundleForm
    statbox_mode = 'create_bundle'

    def process_request(self):
        bundle = Bundle.create_bundle(
            name=self.processed_form.name.data,
            comment=self.processed_form.comment.data,
            created_by=self.validated_uid,
        )
        user_role = UserRole(
            external_type='user',
            role_id=Roles.OWNER.value,
            bundle_uuid=bundle.uuid,
            uid=self.validated_uid,
            created_by=self.validated_uid,
        )
        self.commit(bundle, user_role)

        self.response_values.update({
            'uuid': bundle.uuid,
        })


class UpdateBundleView(BaseBundleView):
    """
    Обновить метаинформацию секрета
    -----
    returns = []
    raises = [NonexistentEntityError]
    example = {
        'arguments': {
            '<string:bundle_uuid>': 'bun-0000000000000000000003t280'
        },
        'data': 'name=passport_top_bundle_1_1&comment=New%20comment&state=normal',
        'response': {
            'status': 'ok'
        }
    }
    """
    form = UpdateBundleForm
    statbox_mode = 'update_bundle'

    def process_request(self, bundle_uuid):
        bundle = Bundle.get_by_id(bundle_uuid)
        self.check_if_has_role(role=Roles.OWNER, bundle_uuid=bundle_uuid)
        bundle = Bundle.update_bundle(
            updated_by=self.validated_uid,
            bundle=bundle,
            name=self.processed_form.name.data,
            comment=self.processed_form.comment.data,
            state=self.processed_form.state.data,
        )
        self.commit(bundle)


class CreateBundleVersionView(BaseBundleView):
    """
    Создать новую версию бандла
    -----
    returns = ['bundle_version']
    raises = [NonexistentEntityError]
    example = {
        'arguments': {
            '<string:bundle_uuid>': 'sec-0000000000000000000000ygj0',
        },
        'headers': {'Content-Type': 'application/json'},
        'data': {
            'secret_versions': [
                '0000000000000000000000ygj6',
                '0000000000000000000000ygj7',
            ]
        },
        'response': {
            'bundle_version': 'ver-0000000000000000000000ygj1',
            'status': 'ok'
        }
    }
    """
    form = CreateBundleVersionForm
    statbox_mode = 'create_bundle_version'

    def process_request(self, bundle_uuid):
        self.check_if_has_role(role=Roles.OWNER, bundle_uuid=bundle_uuid)

        # ToDo: проверять за один присест

        for version_uuid in self.processed_form.secret_versions.entries:
            secret_version = SecretVersion.get_by_id(version_uuid.data)
            self.check_if_has_role(role=Roles.OWNER, secret_uuid=secret_version.secret.uuid)

        bundle_version = BundleVersion.create_bundle_version(
            bundle_uuid=bundle_uuid,
            created_by=self.validated_uid,
            comment=self.processed_form.comment.data,
        )

        # ToDo: Сделать за один коммит

        self.commit(bundle_version)
        secret_versions_links = [{
            'bundle_version': bundle_version.version,
            'secret_version': version_uuid.data,
        } for version_uuid in self.processed_form.secret_versions.entries]

        get_db().session.execute(bundle_version_to_secret_version.insert(values=secret_versions_links))
        self.commit()
        self.response_values.update({
            'bundle_version': bundle_version.version,
        })


class ListBundlesView(BaseBundleView):
    """
    Получить список всех бандлов
    -----
    returns = ['bundles', 'page', 'page_size']
    raises = []
    example = {
        'response': {
             'bundles': [
                 {
                    'acl': [{
                        'created_at': 1445385618.0,
                        'created_by': 100,
                        'creator_login': 'vault-test-100',
                        'login': 'vault-test-100',
                        'role_slug': 'OWNER',
                        'uid': 100,
                    }],
                    'bundle_roles': [{
                        'created_at': 1445385618.0,
                        'created_by': 100,
                        'creator_login': 'vault-test-100',
                        'login': 'vault-test-100',
                        'role_slug': 'OWNER',
                        'uid': 100,
                    }],
                    'comment': 'All secrets 2',
                    'created_at': 1445385618.0,
                    'created_by': 100,
                    'creator_login': 'vault-test-100',
                    'name': 'passport_top_bundle_2',
                    'updated_at': 1445385618.0,
                    'updated_by': 100,
                    'uuid': 'bun-0000000000000000000003t283',
                 },
                 {
                    'acl': [{
                        'created_at': 1445385618.0,
                        'created_by': 100,
                        'creator_login': 'vault-test-100',
                        'login': 'vault-test-100',
                        'role_slug': 'OWNER',
                        'uid': 100
                    }],
                    'bundle_roles': [{
                        'created_at': 1445385618.0,
                            'created_by': 100,
                            'creator_login': 'vault-test-100',
                            'login': 'vault-test-100',
                            'role_slug': 'OWNER',
                            'uid': 100,
                    }],
                    'comment': 'All secrets',
                    'created_at': 1445385618.0,
                    'created_by': 100,
                    'creator_login': 'vault-test-100',
                    'name': 'passport_top_bundle_1',
                    'updated_at': 1445385618.0,
                    'updated_by': 100,
                    'uuid': 'bun-0000000000000000000003t280',
                }
            ],
            'page': 0,
            'page_size': 50,
            'status': 'ok',
        }
    }
    """
    form = ListBundlesForm
    statbox_mode = 'list_bundles'
    use_slave = True

    def fill_acl(self, bundles, abc_ids, staff_ids, uid):
        for bundle in bundles:
            bundle['acl'] = list(filter(
                lambda x: (x.get('uid') == uid or x.get('abc_id') in abc_ids or x.get('staff_id') in staff_ids),
                bundle['bundle_roles'],
            ))

    def process_request(self):
        sorting_column = getattr(Bundle, self.processed_form.order_by.data)
        sorting_column = sorting_column.asc() if self.processed_form.asc.data else sorting_column.desc()

        is_supervisor = self.check_if_supervisor(raises=False)
        filters = and_(
            Bundle.name.like(self.processed_form.starts_with.data + '%'),
            Bundle.state == State.normal.value,
        )

        uid = self.validated_uid

        abc = ExternalRecord.get_abc_groups(uid)
        staff = ExternalRecord.get_staff_groups(uid)

        bundles_query = Bundle.query
        if not is_supervisor:
            filtered_roles = UserRole.get_identity_role_filters(uid=uid)
            bundles_query = bundles_query.join(
                filtered_roles,
                Bundle.uuid == filtered_roles.c.user_roles_bundle_uuid,
            )

        bundles_query = bundles_query.filter(
            filters,
        ).order_by(
            sorting_column,
        ).offset(
            self.processed_form.page.data * self.processed_form.page_size.data,
        ).limit(
            self.processed_form.page_size.data,
        )
        bundles = map(lambda x: x.serialize(max_depth=3, exclude=['bundle_versions']), bundles_query)

        if not is_supervisor:
            self.fill_acl(bundles=bundles, abc_ids=abc, staff_ids=staff, uid=uid)

        self.response_values.update({
            'bundles': list(bundles),
            'page': self.processed_form.page.data,
            'page_size': self.processed_form.page_size.data,
        })
