# -*- coding: utf-8 -*-

import json
import re
import sys
import textwrap

from flask import current_app
from passport.backend.utils.common import filter_none
from passport.backend.vault.api import errors
from passport.backend.vault.api.views import validators
from passport.backend.vault.api.views.base_view import BaseView
import six
from wtforms.fields import FieldList
from wtforms.fields.core import UnboundField


DOCSTRING_RE = re.compile(
    r'''
        ^(.+?)
        (?:\n-{3,}\s*?\n
        (.*))?$
    ''',
    re.X | re.DOTALL,
)


def parse_docstring(view_func):
    """
    Достаем описание и данные из докстринга.
    Данные отделяем от описания строкой с тремя и более дефисами.
    -----
    example = []  # Данные описываем кодом на Питоне
    raises = [AccessError]  # Код выполняем в контексте модуля из которого пришла view_func
    """
    string = view_func.__doc__
    if not string:
        return dict(description='', data={})

    string = textwrap.dedent(string)
    parsed = DOCSTRING_RE.match(string)
    description, code = parsed.group(1, 2) if parsed else ('', '')

    data = dict()
    if code:
        compiled = compile(code, '<string>', 'exec')
        globals = sys.modules[view_func.__module__].__dict__
        globals.update(errors.__dict__)
        exec(compiled, globals, data)

    return dict(
        description=description.strip(),
        data=data,
    )


class DocsView(BaseView):
    """
    Get documentation
    -----
    returns = ['help', 'version']
    """
    allow_origin = '*'
    autodoc = False
    required_user_auth = False

    def validator_to_str(self, x):
        if isinstance(x, (
            validators.DataRequired,
            validators.Regexp,
            validators.AnyOf,
            validators.Length,
        )):
            return x.message
        return None

    def extract_form_help(self, forms):
        result = {}
        if forms:
            if not isinstance(forms, tuple):
                forms = (forms, )
            for form in forms:
                for field in dir(form):
                    if field.startswith('_'):
                        continue

                    field_instance = getattr(form, field)

                    if isinstance(field_instance, tuple):
                        field_instance = field_instance[0]

                    if not isinstance(field_instance, UnboundField):
                        continue

                    description = field_instance.kwargs.get('description')
                    if not description:
                        if isinstance(field_instance.args[0], six.string_types):
                            description = field_instance.args[0]
                            description = description[0].lower() + description[1:]
                            description = description.strip()
                    validators = field_instance.kwargs.get('validators')
                    if validators:
                        validators = filter_none(map(self.validator_to_str, validators))
                    result[field] = {
                        'list': False,
                        'description': description,
                        'validators': validators,
                        'default': field_instance.kwargs.get('default'),
                    }

                    if issubclass(field_instance.field_class, FieldList):
                        result[field].update(dict(
                            list=True,
                            form=self.extract_form_help(
                                # Магия, чтобы добраться до сабформы
                                field_instance.args[0].args[0],
                            ),
                        ))
        return result

    def extract_raises_help(self, raises):
        if raises:
            return list(map(lambda x: x.code, raises))
        return []

    def extract_returns_help(self, returns):
        return ['status'] + list(filter(lambda x: x != 'status', returns))

    def parse_docstring(self, view_func):
        return parse_docstring(view_func)

    def get_doc(self, rule, view_func):
        method = ','.join(set(rule.methods) - {'OPTIONS', 'HEAD'})

        docstring = self.parse_docstring(view_func)
        example = docstring['data'].get('example', dict())
        if 'data' in example:
            if not isinstance(example['data'], str):
                example['data'] = json.dumps(example['data'])
        example['headers'] = example.get('headers', dict())

        raises = list(docstring['data'].get('raises', []))
        raises.extend(view_func.base_common_errors)
        if hasattr(view_func, 'form'):
            raises.extend(view_func.base_form_errors)

        if view_func.required_user_auth:
            example['headers']['X-Ya-User-Ticket'] = '<user_ticket>'
            raises.extend(view_func.base_auth_errors)

        return {
            'form': self.extract_form_help(view_func.form),
            'raises': self.extract_raises_help(raises),
            'returns': self.extract_returns_help(docstring['data'].get('returns', [])),
            'method': method,
            'path': str(rule),
            'host': self.config['application']['balancer'],
            'example': example,
            'description': docstring['description'],
            'arguments': str(','.join(sorted(rule.arguments))),
            'required_user_auth': view_func.required_user_auth,
        }

    def process_request(self, *args, **kwargs):
        arr = []
        for rule in current_app.url_map.iter_rules():
            view_func = current_app.view_functions[rule.endpoint]
            if (
                not hasattr(view_func, 'view_class') or
                not issubclass(view_func.view_class, BaseView) or
                not view_func.view_class.autodoc
            ):
                continue
            arr.append(self.get_doc(rule, view_func))

        self.response_values.update({
            'help': arr,
            'version': self.config['application']['version'],
        })
