# -*- coding: utf-8 -*-

from passport.backend.utils.common import noneless_dict
from passport.backend.vault.api.errors import (
    NonexistentEntityError,
    UserNotFoundError,
)
from passport.backend.vault.api.models import (
    Roles,
    Secret,
    UserInfo,
    UserRole,
)
from passport.backend.vault.api.views.base_view import BaseView
from passport.backend.vault.api.views.roles.forms import SecretUserRoleForm


class BaseRoleView(BaseView):
    def resolve_uid(self, uid=None, login=None):
        try:
            if uid is not None:
                return UserInfo.get_by_id(uid, with_keys=False).uid
            elif login is not None:
                return UserInfo.get_by_login(login, with_keys=False).uid
        except NonexistentEntityError:
            raise UserNotFoundError(uid=uid, login=login)


class CreateSecretUserRoleView(BaseRoleView):
    """
    Добавить новую роль пользователю/группе
    -----
    raises = [AbcScopeNotFoundError, AbcServiceNotFoundError, NonexistentEntityError, StaffGroupNotFoundError, UserNotFoundError]
    returns = []
    example = {
        'arguments': {
            '<string:secret_uuid>': '0000000000000000000000ygj0',
        },
        'data': 'role=READER&staff_id=4112',
        'response': {
            'status': 'ok',
        },
    }
    """
    form = SecretUserRoleForm
    statbox_mode = 'create_secret_user_role'

    def make_user_role(self, secret, form):
        user_role = UserRole.create_user_role(
            creator_uid=self.validated_uid,
            secret=secret,
            role=Roles[form.role.data],
            abc_id=form.abc_id.data,
            abc_scope=form.abc_scope.data,
            abc_role_id=form.abc_role_id.data,
            staff_id=form.staff_id.data,
            uid=self.resolve_uid(form.uid.data, form.login.data),
        )
        return user_role

    def process_request(self, secret_uuid):
        self.check_if_has_role(role=Roles.OWNER, secret_uuid=secret_uuid)
        secret = Secret.get_by_id(secret_uuid)
        added, user_role = UserRole.safely_add_user_role_to_secret(
            self.make_user_role(secret, self.processed_form),
        )
        self.response_status_code = 201 if added else 200

        self.statbox_log(**noneless_dict(dict(
            action='success',
            secret_uuid=str(secret.uuid),
            added=added,
            deleted_by=user_role.created_by,
            uid=user_role.uid or None,
            role=user_role.role.name,
            abc_id=user_role.abc_id or None,
            abc_scope=user_role.abc_scope() or None,
            abc_role_id=user_role.abc_role() or None,
            staff_id=user_role.staff_id or None,
            staff_slug=user_role.staff_slug() or None,
        )))


class DeleteSecretUserRoleView(BaseRoleView):
    """
    Удалить роль у пользователя/группы
    -----
    raises = [AbcScopeNotFoundError, NonexistentEntityError, UserNotFoundError]
    returns = []
    example = {
        'arguments': {
            '<string:secret_uuid>': '0000000000000000000000ygj0',
        },
        'data': 'role=READER&staff_id=4112',
        'response': {
            'status': 'ok',
        },
    }
    """
    form = SecretUserRoleForm
    statbox_mode = 'delete_secret_user_role'

    def delete_user_role_from_secret(self, secret, form):
        role = Roles[form.role.data]
        user_role = UserRole.delete_user_role(
            deleter_uid=self.validated_uid,
            secret=secret,
            role=role,
            abc_id=form.abc_id.data,
            abc_scope=form.abc_scope.data,
            abc_role_id=form.abc_role_id.data,
            staff_id=form.staff_id.data,
            uid=self.resolve_uid(form.uid.data, form.login.data),
        )

        if not user_role:
            self.response_status_code = 204
        self.commit(secret)

        return user_role

    def process_request(self, secret_uuid):
        role = Roles[self.processed_form.role.data]
        need_check_last_owner = (role == Roles.OWNER)

        self.check_if_has_role(role=Roles.OWNER, secret_uuid=secret_uuid, check_if_last=need_check_last_owner)
        secret = Secret.get_by_id(uuid=secret_uuid)

        user_role = self.delete_user_role_from_secret(
            secret=secret,
            form=self.processed_form,
        )

        if user_role is not None:
            self.statbox_log(**noneless_dict(dict(
                action='success',
                secret_uuid=str(secret.uuid),
                deleted_by=self.validated_uid,
                uid=user_role.uid or None,
                role=user_role.role.name,
                abc_id=user_role.abc_id or None,
                abc_scope=user_role.abc_scope() or None,
                abc_role_id=user_role.abc_role() or None,
                staff_id=user_role.staff_id or None,
                staff_slug=user_role.staff_slug() or None,
            )))
