# -*- coding: utf-8 -*-

from passport.backend.utils import common
from passport.backend.vault.api import config
from passport.backend.vault.api.db import get_db
from passport.backend.vault.api.errors import (
    AccessError,
    UserNotFoundError,
)
from passport.backend.vault.api.models import (
    DelegationToken,
    ExternalRecord,
    Roles,
    RolesWeights,
    Secret,
    SecretVersion,
    TvmAppInfo,
    UserInfo,
    UserRole,
)
from passport.backend.vault.api.models.base import (
    MySQLMatch,
    State,
    TokenState,
)
from passport.backend.vault.api.models.tags import (
    Tag,
    TagEntity,
)
from passport.backend.vault.api.utils import ulid
from passport.backend.vault.api.utils.errors import BaseUUIDError
from passport.backend.vault.api.views.base_view import BaseView
from passport.backend.vault.api.views.secrets.forms import (
    CreateSecretForm,
    CreateSupervisorForm,
    GetSecretForm,
    ListSecretsForm,
    UpdateSecretForm,
)
from sqlalchemy import (
    alias,
    and_,
    distinct,
    func,
    or_,
    select,
)
from sqlalchemy.orm import selectinload


class BaseSecretView(BaseView):
    required_user_auth = True

    def fill_acl(self, secret, abc_ids, staff_ids, uid):
        secret['acl'] = list(filter(
            lambda x: (
                x.get('uid') == uid or
                (x.get('abc_id'), x.get('abc_scope_id'), 0) in abc_ids or
                (x.get('abc_id'), 0, x.get('abc_role')) in abc_ids or
                x.get('staff_id') in staff_ids
            ),
            secret['secret_roles'],
        ))
        if secret['acl']:
            secret['effective_role'] = reduce(
                lambda x, y: y['role_slug'] if getattr(RolesWeights, y['role_slug']).value > getattr(RolesWeights, x).value else x,
                secret['acl'],
                min(RolesWeights).name,
            )

    def get_secret(self, secret_uuid, resolve=False):
        if resolve:
            prefix = secret_uuid.split('-', 1)
            prefix = prefix[0] if prefix else ''
            if prefix.lower() == 'ver':
                version = SecretVersion.get_by_id(secret_uuid)
                secret_uuid = version.secret_uuid
        return Secret.get_by_id(secret_uuid)


class ListSecretsView(BaseSecretView):
    """
    Вернуть список всех доступных секретов для пользователя
    -----
    returns = ['secrets', 'page', 'page_size']
    raises = []
    example = {
        'data': '',
        'response': {
            'page': 0,
            'page_size': 50,
            'secrets': [{
                'acl': [{
                    'role_slug': 'OWNER',
                    'uid': 100,
                    'created_at': 1445385604,
                    'created_by': 100,
                    'login': 'vault-test-100',
                    'creator_login': 'vault-test-100',
                }],
                'effective_role': 'OWNER',
                'created_at': 1445385604,
                'created_by': 100,
                'creator_login': 'vault-test-100',
                'name': 'secret_2',
                'last_secret_version': {'version': 'ver-0000000000000000000000ygja'},
                'secret_roles': [{
                    'role_slug': 'OWNER',
                    'uid': 100,
                    'created_at': 1445385604,
                    'created_by': 100,
                    'login': 'vault-test-100',
                    'creator_login': 'vault-test-100',
                }],
                'tokens_count': 0,
                'updated_at': 1445385608,
                'updated_by': 100,
                'uuid': 'sec-0000000000000000000000ygj5',
                'versions_count': 4,
            }, {
                'acl': [{
                    'role_slug': 'OWNER',
                    'uid': 100,
                    'created_at': 1445385600,
                    'created_by': 100,
                    'login': 'vault-test-100',
                    'creator_login': 'vault-test-100',
                }],
                'effective_role': 'OWNER',
                'created_at': 1445385600,
                'created_by': 100,
                'creator_login': 'vault-test-100',
                'name': 'secret_1',
                'last_secret_version': {'version': 'ver-0000000000000000000000ygj4'},
                'secret_roles': [{
                    'role_slug': 'OWNER',
                    'uid': 100,
                    'created_at': 1445385600,
                    'created_by': 100,
                    'login': 'vault-test-100',
                    'creator_login': 'vault-test-100',
                }],
                'tokens_count': 1,
                'updated_at': 1445385600,
                'updated_by': 100,
                'uuid': 'sec-0000000000000000000000ygj0',
                'versions_count': 4,
            }],
            'status': 'ok',
        },
    }
    """
    form = ListSecretsForm
    statbox_mode = 'list_secrets'
    use_slave = True

    def _secrets_expression(self, is_supervisor=False):
        secret_table = alias(Secret)
        from_ = secret_table

        filters = (secret_table.c.state == State.normal.value)
        if self.processed_form.with_hidden_secrets.data:
            filters = or_(
                filters,
                secret_table.c.state == State.hidden.value,
            )

        if self.processed_form.yours.data:
            filters = and_(
                filters,
                secret_table.c.created_by == self.validated_uid,
            )

        tags = Tag.normalize_tags(self.processed_form.tags.data)
        if tags:
            filters = and_(
                filters,
                TagEntity.tags_exists_filter('secret', secret_table.c.uuid, tags),
            )

        if not is_supervisor:
            filtered_roles = UserRole.get_identity_role_filters(
                uid=self.validated_uid,
                roles=Roles[self.processed_form.role.data] if self.processed_form.role.data else None,
            )
            users_secrets = select([
                filtered_roles.c.user_roles_secret_uuid.label('secret_uuid'),
            ]).select_from(
                filtered_roles,
            ).alias('users_secrets')
            from_ = from_.join(
                users_secrets,
                secret_table.c.uuid == users_secrets.c.secret_uuid,
            )

        if self.processed_form.query.data is not None:
            query_string = self.processed_form.query.data.lower().strip()
            query_filters = self._fts_filters(
                secret_table,
                query_string,
                self.processed_form.query_type.data,
            )

            try:
                uuid = ulid.ULID(query_string.strip('-'), ignore_prefix=True)
                query_filters = or_(
                    query_filters,
                    secret_table.c.uuid == uuid,
                    SecretVersion.version == uuid,
                )

                from_ = from_.outerjoin(
                    SecretVersion,
                    and_(
                        secret_table.c.uuid == SecretVersion.secret_uuid,
                        SecretVersion.state == State.normal.value,
                    ),
                )
            except BaseUUIDError:
                pass

            filters = and_(
                filters,
                query_filters,
            )

        expr = select([
            distinct(secret_table.c.uuid).label('secret_uuid'),
        ]).select_from(
            from_,
        ).where(
            filters,
        )

        return expr

    def _fts_filters(self, secret_table, query_string, query_type):
        if query_type == 'language':
            return MySQLMatch(
                [secret_table.c.name, secret_table.c.comment],
                query_string,
                mode=MySQLMatch.NATURAL_MODE,
            )

        if query_type == 'exact':
            return or_(
                secret_table.c.name == query_string,
                secret_table.c.comment == query_string,
            )

        if query_type == 'prefix':
            return or_(
                secret_table.c.name.ilike(query_string + '%'),
                secret_table.c.comment.ilike(query_string + '%'),
            )

        # infix по-умолчанию
        return or_(
            secret_table.c.name.ilike('%' + query_string + '%'),
            secret_table.c.comment.ilike('%' + query_string + '%'),
        )

    def _fetch_tvm_apps(self, secrets_uids):
        result = dict()

        if self.processed_form.with_tvm_apps.data:
            db_session = get_db().session
            secrets_to_apps = list(db_session.execute(
                select(
                    [
                        DelegationToken.secret_uuid.label('secret_uuid'),
                        DelegationToken.tvm_client_id.label('tvm_client_id'),
                    ],
                    distinct=True,
                ).select_from(
                    DelegationToken,
                ).where(
                    and_(
                        DelegationToken.secret_uuid.in_(secrets_uids),
                        DelegationToken.tvm_client_id.isnot(None),
                        DelegationToken.state == TokenState.normal.value,
                    ),
                ).order_by(
                    DelegationToken.secret_uuid,
                    DelegationToken.tvm_client_id,
                ),
            ))
            apps_uuids = set(map(
                lambda x: x['tvm_client_id'],
                secrets_to_apps,
            ))
            if apps_uuids:
                apps = dict(map(
                    lambda x: (x.tvm_client_id, x),
                    TvmAppInfo.query.filter(
                        TvmAppInfo.tvm_client_id.in_(apps_uuids),
                    ),
                ))
                for sa in secrets_to_apps:
                    app = apps.get(sa['tvm_client_id'])
                    if not app:
                        continue
                    result.setdefault(str(sa['secret_uuid']), []).append(app)

        return result

    def process_request(self):
        sorting_column = getattr(Secret, self.processed_form.order_by.data)
        sorting_column = sorting_column.asc() if self.processed_form.asc.data else sorting_column.desc()

        is_supervisor = self.check_if_supervisor(raises=False)
        abc = ExternalRecord.get_abc_groups(self.validated_uid)
        staff = ExternalRecord.get_staff_groups(self.validated_uid)

        db_session = get_db().session

        found_secrets = self._secrets_expression(is_supervisor=is_supervisor).alias('found_secrets')
        secrets = Secret.query.options(
            selectinload(Secret.secret_roles),
            selectinload(Secret.tags_links),
        ).join(
            found_secrets,
            Secret.uuid == found_secrets.c.secret_uuid,
        ).order_by(
            sorting_column,
        ).offset(
            self.processed_form.page.data * self.processed_form.page_size.data,
        ).limit(
            self.processed_form.page_size.data,
        ).all()
        secrets_uids = list(map(lambda x: x.uuid, secrets))

        versions = dict(map(
            lambda x: (str(x[0]), dict(version=x[1], count=x[2]), ),
            db_session.execute(
                select(
                    [
                        SecretVersion.secret_uuid,
                        func.max(SecretVersion.version).label('last_version'),
                        func.count(),
                    ],
                    and_(
                        SecretVersion.secret_uuid.in_(secrets_uids),
                        SecretVersion.state == State.normal.value,
                        SecretVersion.unexpired_version_filter(),
                    ),
                ).group_by(
                    SecretVersion.secret_uuid,
                ),
            ),
        ))

        versions_comments = dict(map(
            lambda x: (str(x[0]), dict(comment=x[1]), ),
            db_session.execute(
                select(
                    [
                        SecretVersion.version,
                        SecretVersion.comment,
                    ],
                    SecretVersion.version.in_([v['version'] for v in versions.values()]),
                ),
            ),
        ))

        secrets_tokens_counts = dict(map(
            lambda x: (str(x[0]), dict(count=x[1]), ),
            db_session.execute(
                select(
                    [
                        DelegationToken.secret_uuid,
                        func.count(),
                    ],
                    and_(
                        DelegationToken.secret_uuid.in_(secrets_uids),
                        DelegationToken.state == TokenState.normal.value,
                    ),
                ).group_by(
                    DelegationToken.secret_uuid,
                ),
            ),

        ))

        tvm_apps = self._fetch_tvm_apps(secrets_uids)

        include_fields = []
        if self.processed_form.with_hidden_secrets.data:
            include_fields.append('state_name')

        serilaized_secrets = map(lambda x: x.serialize(
            max_depth=3,
            exclude=['secret_versions'],
            include=include_fields,
        ), secrets)

        for secret in serilaized_secrets:
            secret_uuid = secret['uuid']
            secret['versions_count'] = versions.get(secret_uuid, {}).get('count', 0)
            secret['tokens_count'] = secrets_tokens_counts.get(secret_uuid, {}).get('count', 0)

            apps = tvm_apps.get(secret_uuid, {})
            if apps:
                secret['tvm_apps'] = map(
                    lambda x: x.serialize(max_depth=3),
                    apps,
                )

            last_version = versions.get(secret_uuid, {}).get('version', None)
            if last_version is not None:
                secret['last_secret_version'] = common.noneless_dict({
                    'version': last_version,
                    'comment': versions_comments.get(str(last_version), {}).get('comment', None),
                })

        if not is_supervisor:
            for secret in serilaized_secrets:
                self.fill_acl(secret=secret, abc_ids=abc, staff_ids=staff, uid=self.validated_uid)

        self.statbox_log(
            action='success',
            results_count=len(secrets),
            **self.processed_form.statbox_data()
        )

        self.response_values.update({
            'secrets': list(serilaized_secrets),
            'page': self.processed_form.page.data,
            'page_size': self.processed_form.page_size.data,
        })


class GetSecretView(BaseSecretView):
    """
    Вернуть секрет и список его версий
    -----
    returns = []
    raises = [NonexistentEntityError]
    example = {
        'arguments': {
            '<string:secret_uuid>': 'sec-0000000000000000000000ygj0',
        },
        'data': '',
        'response': {
            'page': 0,
            'page_size': 50,
            'secret': {
                'acl': [{
                    'role_slug': 'OWNER',
                    'uid': 100,
                    'created_at': 1445385600,
                    'created_by': 100,
                    'login': 'vault-test-100',
                    'creator_login': 'vault-test-100',
                }],
                'effective_role': 'OWNER',
                'created_at': 1445385600,
                'created_by': 100,
                'creator_login': 'vault-test-100',
                'name': 'secret_1',
                'secret_roles': [{
                    'role_slug': 'OWNER',
                    'uid': 100,
                    'created_at': 1445385600,
                    'created_by': 100,
                    'login': 'vault-test-100',
                    'creator_login': 'vault-test-100',
                }],
                'secret_versions': [{
                    'created_at': 1445385603,
                    'created_by': 100,
                    'creator_login': 'vault-test-100',
                    'keys': ['password'],
                    'version': 'ver-0000000000000000000000ygj4',
                }, {
                    'created_at': 1445385602,
                    'created_by': 100,
                    'creator_login': 'vault-test-100',
                    'keys': ['password'],
                    'version': 'ver-0000000000000000000000ygj3',
                }, {
                    'created_at': 1445385601,
                    'created_by': 100,
                    'creator_login': 'vault-test-100',
                    'keys': ['password'],
                    'version': 'ver-0000000000000000000000ygj2',
                }, {
                    'created_at': 1445385600.0,
                    'created_by': 100,
                    'creator_login': 'vault-test-100',
                    'keys': ['cert', 'password'],
                    'version': 'ver-0000000000000000000000nbk1',
                }],
                'tokens': [
                    {
                        'created_at': 1445385600.0,
                        'created_by': 100,
                        'creator_login': 'vault-test-100',
                        'signature': '123',
                        'secret_uuid': 'sec-0000000000000000000000ygj0',
                        'state_name': 'normal',
                        'token_uuid': 'tid-0000000000000000000000nbk0',
                        'tvm_app': {
                            'abc_department': {
                                'display_name': u'\u041f\u0430\u0441\u043f\u043e\u0440\u0442',
                                'id': 14,
                                'unique_name': 'passp'
                            },
                            'abc_state': 'granted',
                            'abc_state_display_name': u'\u0412\u044b\u0434\u0430\u043d',
                            'name': 'social api (dev)',
                            'tvm_client_id': 2000367
                        },
                        'tvm_client_id': 2000367
                    }
                ],
                'updated_at': 1445385600,
                'updated_by': 100,
                'uuid': 'sec-0000000000000000000000ygj0',
            },
            'status': 'ok',
        },
    }
    """
    form = GetSecretForm
    statbox_mode = 'secret'
    use_slave = True

    def validate_access(self, secret):
        is_supervisor = self.check_if_supervisor(raises=False)
        if not is_supervisor:
            try:
                self.check_if_has_role(role=[Roles.READER, Roles.APPENDER], secret_uuid=secret.uuid)
            except AccessError:
                raise AccessError(secret_state_name=secret.state_name())

    def process_request(self, secret_uuid):
        page = self.processed_form.page.data
        page_size = self.processed_form.page_size.data

        secret, secret_versions = Secret.get_secret_with_versions(
            secret_uuid=secret_uuid,
            filters=[SecretVersion.unexpired_version_filter(), ],
            page=page,
            page_size=page_size,
        )

        self.validate_access(secret)

        included_fields = []
        if secret.state != State.normal.value:
            included_fields.append('state_name')

        serialized_secret = secret.serialize(
            max_depth=3,
            include=included_fields,
            exclude=['secret_versions'],
        )
        serialized_secret['secret_versions'] = map(
            lambda x: x.serialize(
                exclude=['value', 'secret_name'],
                include=['parent_version_uuid', 'parent_diff_keys'],
            ),
            secret_versions,
        )
        self.check_state(secret, self.response_values)
        serialized_secret['tokens'] = list(
            map(
                lambda x: x.serialize(max_depth=3),
                secret.tokens.limit(page_size).offset(page * page_size).all(),
            ),
        )

        abc = ExternalRecord.get_abc_groups(self.validated_uid)
        staff = ExternalRecord.get_staff_groups(self.validated_uid)
        self.fill_acl(secret=serialized_secret, abc_ids=abc, staff_ids=staff, uid=self.validated_uid)

        self.response_values.update({
            'page': self.processed_form.page.data,
            'page_size': self.processed_form.page_size.data,
            'secret': serialized_secret,
        })


class CreateSecretView(BaseSecretView):
    """
    Создать новый секрет
    -----
    returns = ['uuid']
    raises = [UserNotFoundError]
    example = {
        'data': 'name=passport_top_secret_1&comment=Dont%2Ftouch&tags=yp,search',
        'response': {
            'uuid': 'sec-0000000000000000000000ygj0',
            'status': 'ok',
        },
    }
    """
    form = CreateSecretForm
    statbox_mode = 'create_secret'

    def create_secret(self, form):
        secret = Secret.create_secret(
            created_by=self.validated_uid,
            name=form.name.data,
            comment=form.comment.data,
            tags=form.tags.data,
        )
        user_role = UserRole.create_user_role(
            creator_uid=self.validated_uid,
            secret=secret,
            role=Roles.OWNER,
            uid=self.validated_uid,
            validate_uid=config['application']['validate_uid_on_secret_creation'],
        )
        self.statbox_log(
            action='success',
            secret_uuid=str(secret.uuid),
            name=secret.name,
            created_by=secret.created_by,
        )
        self.add(secret, user_role)
        self.response_values.update({
            'uuid': secret.uuid.str(),
        })
        return secret

    def process_request(self):
        self.create_secret(form=self.processed_form)
        self.commit()


class UpdateSecretView(BaseSecretView):
    """
    Обновить метаинформацию секрета
    -----
    returns = []
    raises = [NonexistentEntityError]
    example = {
        'arguments': {
            '<string:secret_uuid>': 'sec-0000000000000000000000ygj0',
        },
        'data': 'name=passport_top_secret_1&comment=Dont%2Ftouch&tags=yp,search&state=normal',
        'response': {
            'status': 'ok',
        },
    }
    """
    form = UpdateSecretForm
    statbox_mode = 'update_secret'

    def process_request(self, secret_uuid):
        secret = self.get_secret(secret_uuid)
        self.check_if_has_role(role=Roles.OWNER, secret_uuid=secret_uuid)
        secret = Secret.update_secret(
            updated_by=self.validated_uid,
            secret=secret,
            name=self.processed_form.name.data,
            comment=self.processed_form.comment.data,
            tags=self.processed_form.tags.data,
            state=self.processed_form.state.data,
        )
        self.statbox_log(
            action='success',
            secret_uuid=str(secret.uuid),
            name=secret.name,
            created_by=secret.created_by,
            state=secret.state_name(),
        )
        self.commit(secret)


class CreateSupervisorView(BaseSecretView):
    """
    Добавить роль SUPERVISOR пользователю
    -----
    raises = []
    returns = []
    example = {
        'data': 'uid=100500',
        'response': {
            'status': 'ok',
        },
    }
    """
    form = CreateSupervisorForm
    statbox_mode = 'create_supervisor'

    def process_request(self):
        self.check_if_supervisor(raises=True)
        user_role = UserRole(
            external_type='user',
            role_id=Roles.SUPERVISOR.value,
            uid=self.processed_form.uid.data,
            created_by=self.validated_uid,
        )
        self.commit(user_role)


class GetOwnersView(BaseSecretView):
    """
    Получить список владельцев секретов
    -----
    returns = ['owners', 'secret_uuid', 'name']
    raises = [NonexistentEntityError]
    example = {
        'arguments': {'<string:secret_uuid>': 'sec-0000000000000000000000ygj0'},
        'response': {
            'secret_uuid': u'sec-0000000000000000000000ygj0',
            'name': 'secret_1',
            'owners': [{
                'created_at': 1445385600.0,
                'created_by': 100,
                'creator_login': 'vault-test-100',
                'login': 'vault-test-100',
                'role_slug': 'OWNER',
                'uid': 100,
            }],
            'status': 'ok',
        },
    }
    """
    statbox_mode = 'owners'
    use_slave = True
    required_user_auth = False

    def process_request(self, secret_uuid):
        secret = self.get_secret(secret_uuid, resolve=True)
        self.check_state(secret, self.response_values)

        owners = filter(lambda x: x.role_id == Roles.OWNER, secret.secret_roles)
        self.response_values.update({
            'secret_uuid': secret.uuid,
            'name': secret.name,
            'owners': owners,
        })


class GetReadersView(BaseSecretView):
    """
    Получить список читателей секретов
    -----
    returns = ['readers', 'secret_uuid', 'name']
    raises = [NonexistentEntityError]
    example = {
        'arguments': {'<string:secret_uuid>': 'sec-0000000000000000000000ygj0'},
        'response': {
            'secret_uuid': u'sec-0000000000000000000000ygj0',
            'name': 'secret_1',
            'readers': [{
                'created_at': 1445385600.0,
                'created_by': 100,
                'creator_login': 'vault-test-100',
                'login': 'vault-test-100',
                'role_slug': 'OWNER',
                'uid': 100,
            }],
            'status': 'ok',
        },
    }
    """
    statbox_mode = 'readers'
    use_slave = True
    required_user_auth = False

    def process_request(self, secret_uuid):
        secret = self.get_secret(secret_uuid, resolve=True)
        self.check_state(secret, self.response_values)

        readers = filter(
            lambda x: (x.role_id == Roles.OWNER or x.role_id == Roles.READER),
            secret.secret_roles,
        )
        self.response_values.update({
            'secret_uuid': secret.uuid,
            'name': secret.name,
            'readers': readers,
        })


class GetWritersView(BaseSecretView):
    """
    Получить список пользователей, которые могут менять секрет
    -----
    returns = ['writers', 'secret_uuid', 'name']
    raises = [NonexistentEntityError]
    example = {
        'arguments': {'<string:secret_uuid>': 'sec-0000000000000000000000ygj0'},
        'response': {
            'secret_uuid': u'sec-0000000000000000000000ygj0',
            'name': 'secret_1',
            'writers': [{
                'created_at': 1445385600.0,
                'created_by': 100,
                'creator_login': 'vault-test-100',
                'login': 'vault-test-100',
                'role_slug': 'OWNER',
                'uid': 100,
            }],
            'status': 'ok',
        },
    }
    """
    statbox_mode = 'writers'
    use_slave = True
    required_user_auth = False

    def process_request(self, secret_uuid):
        secret = self.get_secret(secret_uuid, resolve=True)
        self.check_state(secret, self.response_values)

        writers = filter(
            lambda x: (x.role_id == Roles.OWNER or x.role_id == Roles.APPENDER),
            secret.secret_roles,
        )
        self.response_values.update({
            'secret_uuid': secret.uuid,
            'name': secret.name,
            'writers': writers,
        })


class CheckUserCanReadSecretView(BaseSecretView):
    """
    Проверить, что пользователь может читать секрет
    -----
    returns = ['secret_uuid', 'name', 'user', 'access']
    raises = [NonexistentEntityError, UserNotFoundError]
    example = {
        'arguments': {
            '<string:secret_uuid>': 'sec-0000000000000000000000ygj0',
            '<string:uid>': '100',
        },
        'response': {
            'secret_uuid': u'sec-0000000000000000000000ygj0',
            'name': 'secret_1',
            'access': 'allowed',
            'status': 'ok',
            'user': {
                'login': 'vault-test-100',
                'uid': 100,
            },
        },
    }
    """
    statbox_mode = 'check_user_can_read_secret'
    use_slave = True
    required_user_auth = False

    def process_request(self, secret_uuid, uid):
        secret = self.get_secret(secret_uuid, resolve=True)

        user = UserInfo.query.filter(UserInfo.uid == uid).one_or_none()
        if user is None:
            raise UserNotFoundError(uid)

        can_read_secret = self.check_if_has_role(
            role=[Roles.READER, Roles.OWNER],
            secret_uuid=secret.uuid,
            uid=user.uid,
            raises=False,
        )

        self.response_values.update({
            'secret_uuid': secret.uuid,
            'name': secret.name,
            'user': {
                'uid': user.uid,
                'login': user.login,
            },
            'access': 'allowed' if can_read_secret else 'denied',
        })
