# coding: utf-8

from passport.backend.vault.api.models import (
    AbcDepartmentInfo,
    StaffDepartmentInfo,
    TvmAppInfo,
    UserInfo,
)
from passport.backend.vault.api.views.base_view import BaseView
from passport.backend.vault.api.views.suggest.forms import (
    SuggestForm,
    TvmSuggestForm,
)


class BaseSuggestView(BaseView):
    autodoc = True
    required_user_auth = False


class SuggestView(BaseSuggestView):
    """
    Поиск по хранящимся в системе объектам
    -----
    returns = ['abc_departments', 'staff_departments', 'users']
    raises = []
    example = {
        'data': 'query=Пер',
        'response': {
            'abc_departments': [{
                'display_name': u'\u041f\u0435\u0440\u0435\u0432\u043e\u0434 \u0441\u0430\u0434\u0436\u0435\u0441\u0442\u0430',
                'id': 50,
                'unique_name': 'suggest',
                'roles': [
                    {
                        u'display_name': u'\u0420\u0430\u0437\u0440\u0430\u0431\u043e\u0442\u0447\u0438\u043a',
                        u'id': 8,
                    },
                    {
                        u'display_name': u'\u0421\u0438\u0441\u0442\u0435\u043c\u043d\u044b\u0439 \u0430\u0434\u043c\u0438\u043d\u0438\u0441\u0442\u0440\u0430\u0442\u043e\u0440',
                        u'id': 16
                    },
                ],
                'scopes': [
                    {
                        'display_name': u'\u0410\u0434\u043c\u0438\u043d\u0438\u0441\u0442\u0440\u0438\u0440\u043e\u0432\u0430\u043d\u0438\u0435',
                        'unique_name': u'administration',
                    },
                    {
                        'display_name': u'\u0420\u0430\u0437\u0440\u0430\u0431\u043e\u0442\u043a\u0430',
                        u'unique_name': u'development'
                    },
                ],
            }],
            'staff_departments': [],
            'status': 'ok',
            'users': [{
                'first_name': u'\u041f\u0430\u0448\u0430',
                'last_name': u'\u041f\u0435\u0440\u0435\u0432\u0435\u0434\u0435\u043d\u0446\u0435\u0432',
                'login': 'ppodolsky',
                'uid': 1120000000038274,
                'staff_info': {
                    u'display_name': u'\u0413\u0440\u0443\u043f\u043f\u0430 \u0430\u043d\u0430\u043b\u0438\u0442\u0438\u043a\u0438 '
                                     u'\u0438 \u0431\u0435\u0437\u043e\u043f\u0430\u0441\u043d\u043e\u0441\u0442\u0438 '
                                     u'\u0441\u0438\u0441\u0442\u0435\u043c \u0430\u0432\u0442\u043e\u0440\u0438\u0437\u0430\u0446\u0438\u0438',
                    u'id': 2864,
                    u'unique_name': u'yandex_personal_com_aux_sec',
                },
            }]
        }
    }
    """
    form = SuggestForm
    use_slave = True

    def process_abc_query(self, query, limit=None):
        self.response_values.update({
            'abc_departments': map(
                lambda x: x.serialize(
                    max_depth=3,
                    include=['scopes', 'roles'],
                ),
                AbcDepartmentInfo.get_by_query(query, limit),
            )
        })

    def process_staff_query(self, query, limit=None):
        self.response_values.update({
            'staff_departments': StaffDepartmentInfo.get_by_query(query, limit),
        })

    def process_user_query(self, query, limit=None):
        self.response_values.update({
            'users': map(
                lambda x: x.serialize(
                    max_depth=2,
                    include=['staff_info'],
                ),
                UserInfo.get_by_query(query, limit),
            )
        })

    def process_request(self):
        query = self.processed_form.query.data
        limit = self.processed_form.limit.data

        self.process_abc_query(query, limit)
        self.process_staff_query(query, limit)
        self.process_user_query(query, limit)


class TvmSuggestView(BaseSuggestView):
    """
    Поиск по TVM-приложениям
    -----
    returns = ['tvm_apps']
    raises = []
    example = {
        'data': 'query=passp',
        'response': {
            'status': 'ok',
            'tvm_apps': [
                {
                    u'abc_department': {
                        u'display_name': u'\u041f\u0430\u0441\u043f\u043e\u0440\u0442',
                        u'id': 14,
                        u'unique_name': u'passp',
                    },
                    u'abc_state': u'granted',
                    u'abc_state_display_name': u'\u0412\u044b\u0434\u0430\u043d',
                    u'name': u'passport_likers3',
                    u'tvm_client_id': 2000355,
                },
                {
                    u'abc_department': {
                        u'display_name': u'\u041f\u0430\u0441\u043f\u043e\u0440\u0442',
                        u'id': 14,
                        u'unique_name': u'passp',
                    },
                    u'abc_state': u'granted',
                    u'abc_state_display_name': u'\u0412\u044b\u0434\u0430\u043d',
                    u'name': u'push-client-passport',
                    u'tvm_client_id': 2000230,
                },
            ]
        }
    }
    """
    form = TvmSuggestForm
    use_slave = True

    def process_request(self):
        query = self.processed_form.query.data
        abc_state = self.processed_form.abc_state.data
        limit = self.processed_form.limit.data

        self.response_values.update({
            'tvm_apps': map(
                lambda x: x.serialize(max_depth=4),
                TvmAppInfo.get_by_query(query, abc_state, limit),
            ),
        })
