# coding: utf-8

from passport.backend.vault.api.models import (
    Secret,
    Tag,
    TagEntity,
    UserRole,
)
from passport.backend.vault.api.models.base import State
from passport.backend.vault.api.views.base_view import BaseView
from passport.backend.vault.api.views.tags.forms import SuggestTagsForm
from sqlalchemy import and_


class BaseTagsView(BaseView):
    use_slave = True


class ListTagsView(BaseTagsView):
    """
    Получить список тэгов для пользователя
    -----
    returns = ['tags']
    raises = []
    example = {
        'data': '',
        'response': {
            'tags': [],
            'status': 'ok',
        }
    }
    """
    def get_tags_for_user(self):
        filtered_roles = UserRole.get_identity_role_filters(uid=self.validated_uid)

        tags = Tag.query.distinct().join(
            TagEntity,
        ).join(
            filtered_roles,
            TagEntity.tag_entity_join_filters('secret', filtered_roles.c.user_roles_secret_uuid),
        ).join(
            Secret,
            and_(
                Secret.uuid == filtered_roles.c.user_roles_secret_uuid,
                Secret.state == State.normal.value,
            )
        ).order_by(
            TagEntity.tag,
        )
        return sorted([tag.title for tag in tags])

    def process_request(self):
        self.response_values.update({
            'tags': self.get_tags_for_user(),
        })


class SuggestTagsView(BaseTagsView):
    """
    Саджест тэгов
    -----
    returns = ['tags', 'text', 'page', 'page_size']
    raises = []
    example = {
        'data': 'text=fir',
        'response': {
            'page': 0,
            'page_size': 50,
            'status': 'ok',
            'tags': [],
            'text': 'fir'
        }
    }
    """
    required_user_auth = False
    form = SuggestTagsForm

    def process_request(self):
        tags = Tag.find_tags(
            self.processed_form.text.data,
            limit=self.processed_form.page_size.data,
            offset=self.processed_form.page.data * self.processed_form.page_size.data,
        )

        self.response_values.update({
            'page': self.processed_form.page.data,
            'page_size': self.processed_form.page_size.data,
            'status': 'ok',
            'tags': [tag.title for tag in tags] or [],
            'text': self.processed_form.text.data,
        })
